import { Injectable, OnDestroy } from '@angular/core';
import { Observable, BehaviorSubject, combineLatest } from 'rxjs';
import { CalendarDate } from '../../core/calendar/calendar-date';
import { CalendarDateBuilder } from '../../core/calendar/calendar-date-builder';
import { RoomService } from '../room/room.service';
import { map, tap, first, debounceTime } from 'rxjs/operators';
import { Property } from '../../core/room/property';
import { CalendarItem } from '../../core/calendar/calendar-item';
import { RoomDropZone } from '../../core/calendar/room-drop-zone';
import { CalendarSortOrder } from '../../core/filters/calendar-sort-order';
import { CalendarBlockItem } from '../../core/calendar/calendar-block-item';
import { Conflict } from '../../core/room/conflict';
import { SubSink } from 'subsink';
import { SettingsManager } from '../settings/settings-manager';
import { ScrollPosition } from '../../core/calendar/scroll-position';

type FontClass = 'large' | 'medium' | 'small' | 'xsmall';

@Injectable()
export class CalendarService implements OnDestroy {
  /**
   * The starting column of the date columns. One column for the room,
   * one column for the accommodation type, and two columns for the
   * previous navigation.
   */
  static DATE_COLUMN_START = 7;

  /**
   * The column template for the grid based on the current period.
   */
  gridColumnTemplate$: Observable<string>;

  /**
   * The grid column for the nav next buttons on the calendar.
   */
  navNextGridColumn$: Observable<number>;

  /**
   * The dates that are part of the current period.
   */
  dates$: Observable<CalendarDate[]>;

  /**
   * The start columns by date for the current period.
   */
  private startColumnsByDate$: Observable<{[key: string]: number}>;

  /**
   * The mid columns by date for the current period.
   */
  private midColumnsByDate$: Observable<{[key: string]: number}>;

  /**
   * The properties to be rendered on the calendar.
   */
  properties$: Observable<Property[]>;

  /**
   * The source for rowByAccommodation$.
   */
  private accommodationRowsSource: BehaviorSubject<{[key: string]: number}>;

  /**
   * The mapping of accommodation ids to rows.
   */
  accommodationRows$: Observable<{[key: string]: number}>;

  /**
   * The source for rowByRoomId$.
   */
  private roomRowsSource: BehaviorSubject<{[key: string]: number}>;

  /**
   * The mapping of room ids to rows.
   */
  roomRows$: Observable<{[key: string]: number}>;

  /**
   * The items to be rendered.
   */
  items$: Observable<CalendarItem[]>;

  /**
   * The block items to be rendered.
   */
  blockItems$: Observable<CalendarBlockItem[]>;

  /**
   * The drop zones in the room section.
   */
  roomDropZones$: Observable<RoomDropZone[]>;

  /**
   * The calendar sort order.
   */
  sortBy$: Observable<CalendarSortOrder>;

  /**
   * The font class that the calendar should render in.
   */
  fontClass$: Observable<FontClass>;

  /**
   * The mapping of reservations to colours.
   */
  reservationColours$: Observable<{[key: string]: string}>;

  /**
   * The focused search result item
   */
  focusedItem$: Observable<string|null>;

  /**
   * List of items that match search criteria
   */
  sortedResults$: Observable<string[]>;

  /**
   * The source for activeItem$.
   */
  activeItemSource: BehaviorSubject<CalendarItem|null>;

  /**
   * The item that is current active on the calendar.
   */
  activeItem$: Observable<CalendarItem|null>;

  /**
   * The source for viewItem$
   */
  private viewItemSource: BehaviorSubject<CalendarItem|null>;

  /**
   * The item currently being viewed in the sidenav
   */
  viewItem$: Observable<CalendarItem|null>;

  /**
   * Emits when the rooming calendar is loading or finished loading
   */
  loading$: Observable<boolean>;

  /**
   * The rooming conflicts for the current configuration.
   */
  conflicts$: Observable<Conflict[]>;

  /**
   * The source for conflictIndex$.
   */
  private conflictIndexSource: BehaviorSubject<number>;

  /**
   * The index of the current conflict.
   */
  conflictIndex$: Observable<number>;

  /**
   * The source for conflictItemIndex$.
   */
  private conflictItemIndexSource: BehaviorSubject<number>;

  /**
   * The index of the current conflict item.
   */
  conflictItemIndex$: Observable<number>;

  /**
   * The total number of conflicts there are.
   */
  totalConflicts$: Observable<number>;

  /**
   * The total number of items in the current conflict.
   */
  totalConflictItems$: Observable<number>;

  private subsink: SubSink;

  /**
   * The current conflict that is being viewed.
   */
  selectedConflict$: Observable<Conflict|null>;

  /**
   * The conflict item that is currently selected.
   */
  selectedConflictItem$: Observable<CalendarItem|null>;

  /**
   * The source for conflictsDismissed$.
   */
  private conflictsDismissedSource: BehaviorSubject<boolean>;

  /**
   * Whether the conflicts have been dismissed or not.
   */
  conflictsDismissed$: Observable<boolean>;

  /**
   * The source for calendarScrolled$.
   */
  calendarScrolledSource: BehaviorSubject<ScrollPosition>;

  /**
   * The position that the calendar was scrolled to.
   */
  calendarScrolled$: Observable<ScrollPosition>;

  /**
   * Whether the conflict resolution was interacted with. This allows
   * us to determine whether we want to scroll to conflict items or not
   * when things change.
   */
  private hasConflictInteraction: boolean;

  /**
   * The source for legacyItem$.
   */
  private sidenavItemSource: BehaviorSubject<CalendarItem|null>;

  /**
   * The item that the sidenav was interacting with.
   */
  sidenavItem$: Observable<CalendarItem|null>;


  /**
   * Create the calendar service.
   * @param roomService The room serve.
   */
  constructor(private roomService: RoomService) {
    this.gridColumnTemplate$ = this.gridColumnTemplate();
    this.navNextGridColumn$ = this.navNextGridColumn();
    this.dates$ = this.dates();
    this.fontClass$ = this.fontClass();
    this.startColumnsByDate$ = this.startColumnsByDate();
    this.midColumnsByDate$ = this.midColumnsByDate();
    this.properties$ = roomService.properties$;
    this.sortBy$ = roomService.sortBy$;
    this.accommodationRowsSource = new BehaviorSubject<{[key: string]: number}>({});
    this.accommodationRows$ = this.accommodationRowsSource.asObservable();
    this.roomRowsSource = new BehaviorSubject<{[key: string]: number}>({});
    this.roomRows$ = this.roomRowsSource.asObservable();
    this.buildRowMappings();
    this.items$ = roomService.visibleItems$;
    this.blockItems$ = roomService.visibleBlockItems$;
    this.reservationColours$ = this.reservationColours();
    this.roomDropZones$ = roomService.roomDropZones$;
    this.focusedItem$ = combineLatest([roomService.resultIndex$, roomService.sortedResults$]).pipe(
      map(([index, result]) => {
        if (index >= 0) {
          return result[index];
        } else {
          return null;
        }
      }),
      tap(id => {
        if (id) {
          this.scrollToItem(id);
        }
      })
    );
    this.activeItemSource = new BehaviorSubject<CalendarItem>(null);
    this.activeItem$ = this.activeItemSource.asObservable();
    this.loading$ = roomService.loading$;
    this.sortedResults$ = roomService.sortedResults$;
    this.viewItemSource = new BehaviorSubject<CalendarItem>(null);
    this.viewItem$ = this.viewItemSource.asObservable();
    this.conflicts$ = this.roomService.conflicts$;
    this.conflictIndexSource = new BehaviorSubject<number>(0);
    this.conflictIndex$ = this.conflictIndexSource.asObservable();
    this.conflictItemIndexSource = new BehaviorSubject<number>(0);
    this.conflictItemIndex$ = this.conflictItemIndexSource.asObservable();
    this.sidenavItemSource = new BehaviorSubject<CalendarItem|null>(null);
    this.sidenavItem$ = this.sidenavItemSource.asObservable();
    this.subsink = new SubSink();

    this.subsink.sink = this.items$.subscribe(() => {
      this.hasConflictInteraction = false;
      const sidenavItem = this.sidenavItemSource.value;

      if (sidenavItem) {
        let updatedItem: CalendarItem = null;

        if (sidenavItem.data.id) {
          // If the item had a room group id then find that specific item.
          updatedItem = this.roomService.itemByGroupId(sidenavItem.data.id);
        } else {
          // If the item was not allocated then find the first one that isn't allocated.
          const items = this.roomService.itemsByReservationId(sidenavItem.data.reservationId)
            .filter(item => !item.data.id);

          if (items.length > 0) {
            updatedItem = items[0];
          }
        }

        this.activeItemSource.next(updatedItem);
        this.sidenavItemSource.next(updatedItem);
      } else {
        this.activeItemSource.next(null);
      }
    });

    this.subsink.sink = this.activeItem$.subscribe(item => {
      if (item) {
        this.findDropZones(item);
      }
    });

    this.subsink.sink = this.conflicts$.subscribe(() => {
      this.conflictIndexSource.next(0);
      this.conflictItemIndexSource.next(0);
    });

    this.subsink.sink = this.conflictIndex$.subscribe(() => {
      this.conflictItemIndexSource.next(0);
    });

    this.totalConflicts$ = this.conflicts$.pipe(map(conflicts => conflicts.length));

    this.totalConflictItems$ = combineLatest([this.conflicts$, this.conflictIndex$]).pipe(
      map(([conflicts, index]) => {
        if (conflicts.length > 0) {
          return conflicts[index].items.length;
        } else {
          return 0;
        }
      })
    );

    this.selectedConflict$ = combineLatest([this.conflicts$, this.conflictIndex$]).pipe(
      map(([conflicts, index]) => {
        if (conflicts.length > 0) {
          return conflicts[index];
        } else {
          return null;
        }
      })
    );

    this.selectedConflictItem$ = combineLatest([
      this.selectedConflict$,
      this.conflictItemIndex$
    ]).pipe(
      map(([conflict, index]) => {
        const item = conflict?.items[index];

        if (item) {
          return item;
        } else {
          return null;
        }
      })
    );

    const settingsManager = new SettingsManager();
    const showConflictsOnLoad = settingsManager.getLastSetting('showConflictsOnLoad');
    this.conflictsDismissedSource = new BehaviorSubject<boolean>(!showConflictsOnLoad);
    this.conflictsDismissed$ = this.conflictsDismissedSource.asObservable();

    this.subsink.sink = this.roomService.items$.subscribe(() => {
      const showConflictsOnLoad = settingsManager.getLastSetting('showConflictsOnLoad');
      this.onConflictsDismissedChange(!showConflictsOnLoad);
    });

    this.subsink.sink = combineLatest([this.selectedConflictItem$, this.conflictsDismissed$]).subscribe(([item, dismissed]) => {
      if (item && !dismissed && this.hasConflictInteraction) {
        this.onItemInteraction(item);
        this.scrollToItem(item.id, 350);
      }
    });

    const scrollPosition = settingsManager.getScrollPosition();
    this.calendarScrolledSource = new BehaviorSubject<ScrollPosition>(scrollPosition);
    this.calendarScrolled$ = this.calendarScrolledSource.asObservable().pipe(debounceTime(1000));

    this.subsink.sink = this.calendarScrolled$.subscribe(scroll => {
      settingsManager.setScrollPosition(scroll);
    });

    this.subsink.sink = this.roomService.roomingChanges$.subscribe(() => {
      if (this.activeItemSource.value) {
        this.roomService.findDropZones(this.activeItemSource.value);
      }
    });
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  /**
   * The grid column template based on the current period.
   */
  private gridColumnTemplate(): Observable<string> {
    return combineLatest(this.roomService.period$, this.roomService.sortBy$).pipe(
      map(([period, sortBy]) => {
        /**
         * The first four columns are the search header.
         * The navigate to previous widget takes up two more columns.
         */

        let columnTemplate = '55px 55px 55px 55px 50px 50px';

        /**
         * Each day adds two columns to allow for showing arrival and departures.
         */
        for (let i = 0; i < period.days; i++) {
          columnTemplate += ' minmax(30px, 1fr) minmax(30px, 1fr)';
        }

        /**
         * The navigate to next widget takes up the last two columns.
         */
        columnTemplate += ' 50px 50px';
        return columnTemplate;
      })
    );
  }

  /**
   * The grid column for the nav next component based on the current
   * period.
   */
  private navNextGridColumn(span: number = 1): Observable<number> {
    return this.roomService.period$.pipe(
      map(period => {
        return CalendarService.DATE_COLUMN_START + period.days * 2;
      })
    );
  }

  /**
   * The calendar dates for the current period.
   */
  private dates(): Observable<CalendarDate[]> {
    return this.roomService.period$.pipe(
      map(period => {
        return CalendarDateBuilder.calendarDates(period.start, period.days);
      })
    );
  }

  /**
   * Determines the font class to be used based on the number of days
   * being displayed.
   */
  private fontClass(): Observable<FontClass> {
    return this.roomService.period$.pipe(
      map(period => {
        if (period.days <= 7) {
          return 'large';
        } else if (period.days <= 14) {
          return 'medium';
        } else if (period.days <= 21) {
          return 'small';
        } else {
          return 'xsmall';
        }
      })
    );
  }

  /**
   * Builds up the date to start column map.
   */
  private startColumnsByDate(): Observable<{[key: string]: number}> {
    return this.dates$.pipe(
      map(dates => {
        const startColumnsByDate = {};

        for (let i = 0; i < dates.length; i++) {
          startColumnsByDate[dates[i].date] = CalendarService.DATE_COLUMN_START + i * 2;
        }

        return startColumnsByDate;
      })
    );
  }

  /**
   * Builds up the date to mid column map.
   */
  private midColumnsByDate(): Observable<{[key: string]: number}> {
    return this.dates$.pipe(
      map(dates => {
        const startColumnsByDate = {};

        for (let i = 0; i < dates.length; i++) {
          startColumnsByDate[dates[i].date] = CalendarService.DATE_COLUMN_START + i * 2 + 1;
        }

        return startColumnsByDate;
      })
    );
  }

  /**
   * Retrieves the start column for the given date.
   * @param date The date to determine the column for.
   */
  startColumnForDate(date: string): Observable<number> {
    return this.startColumnsByDate$.pipe(
      map(dates => {
        if (date in dates) {
          return dates[date];
        } else {
          return NaN;
        }
      })
    );
  }

  /**
   * Calculates the column for the middle of the given date based on the
   * current period.
   * @param date The date to determine the column for.
   */
  midColumnForDate(date: string): Observable<number> {
    return this.midColumnsByDate$.pipe(
      map(dates => {
        if (date in dates) {
          return dates[date];
        } else {
          return NaN;
        }
      })
    );
  }

  /**
   * Builds the row mappings for properties, accommodations, rooms,
   * virtual rooms, and property separators.
   */
  private buildRowMappings(): void {
    this.roomService.properties$.subscribe(properties => {
      const accommodationRows = {};
      const roomRows = {};

      for (const property of properties) {
        let row = 1;

        if (property.showAvailability) {
          for (const accommodation of property.accommodations) {
            accommodationRows[accommodation.id] = row++;
          }
        }

        for (const room of property.rooms) {
          roomRows[room.id] = row++;
        }
      }

      this.accommodationRowsSource.next(accommodationRows);
      this.roomRowsSource.next(roomRows);
    });
  }

  /**
   * Determines the row for the accommodation.
   * @param accommodationId The id of the accommodation.
   */
  accommodationRow(propertyId: string, accommodationId: string): Observable<number> {
    return this.accommodationRows$.pipe(
      map(accommodationRows => {
        if (accommodationId in accommodationRows) {
          return accommodationRows[accommodationId];
        } else {
          return NaN;
        }
      })
    );
  }

  /**
   * Determines the row for the room.
   * @param roomId The id of the room.
   */
  roomRow(roomId: string): Observable<number> {
    return this.roomRows$.pipe(
      map(roomRows => {
        if (roomId in roomRows) {
          return roomRows[roomId];
        } else {
          return NaN;
        }
      })
    );
  }

  private generateColours(n: number): string[] {
    const colours = [];
    let hue: number;
    let saturation: number;
    let lightness: number;

    for (let i = 0; i < 360; i += 360 / n) {
      hue = i;
      saturation = 50 + Math.random() * 10;
      lightness = 30 + Math.random() * 10;
      colours.push(`hsl(${hue}, ${saturation}%, ${lightness}%)`);
    }
    return colours;
  }

  private reservationColours(): Observable<{[key: string]: string}> {
    return this.roomService.items$.pipe(
      map(items => {
        const colourMap = {};

        items.forEach(item => {
          if (!(item.data.reservationId in colourMap)) {
            colourMap[item.data.reservationId] = '';
          }
        });

        const reservationIds = Object.keys(colourMap);
        const colours = this.generateColours(reservationIds.length);

        for (let i = 0; i < colours.length; i++) {
          colourMap[reservationIds[i]] = colours[i];
        }

        return colourMap;
      })
    );
  }

  /**
   * Retrieves the colour assigned to a reservation.
   * @param id The id of the reservation.
   */
  reservationColour(id: string): Observable<string> {
    return this.reservationColours$.pipe(
      map(colourMap => {
        if (id in colourMap) {
          return colourMap[id];
        } else {
          return '#a0a0a0';
        }
      })
    );
  }

  /**
   * Finds possible drop zones for the calendar item.
   */
  findDropZones(item: CalendarItem): void {
    this.roomService.findDropZones(item);
  }

  /**
   * Called when an item was released on a drop zone.
   * @param zone The zone where the item was dropped.
   * @param item The item that was moved.
   */
  itemDropped(zone: RoomDropZone, item: CalendarItem): void {
    this.roomService.itemDropped(zone, item);
  }

  private scrollToItem(id: string, offset: number = 250): void {
    const roomItem = document.getElementById(id);
    if (roomItem) {
      const bodyRect = document.body.getBoundingClientRect().top;
      const elementRect = roomItem.getBoundingClientRect().top;
      const elementPosition = elementRect - bodyRect;
      const offsetPosition = elementPosition - offset;

      window.scrollTo({
        top: offsetPosition,
      });
    }
  }

  onItemInteraction(item: CalendarItem): void {
    this.activeItemSource.next(item);
  }

  cancelItemInteraction(): void {
    this.activeItemSource.next(null);
  }

  viewActiveItem() {
    const item = this.activeItemSource.value;
    if (item) {
      this.viewItem(item);
    }
  }

  viewItem(item: CalendarItem) {
    this.viewItemSource.next(item);
  }

  async onNextConflict(): Promise<void> {
    this.hasConflictInteraction = true;
    const index = this.conflictIndexSource.value;
    const total = await this.totalConflicts$.pipe(first()).toPromise();

    if (index + 1 === total) {
      this.conflictIndexSource.next(0);
    } else {
      this.conflictIndexSource.next(index + 1);
    }
  }

  async onPreviousConflict(): Promise<void> {
    this.hasConflictInteraction = true;
    const index = this.conflictIndexSource.value;
    const total = await this.totalConflicts$.pipe(first()).toPromise();

    if (index === 0) {
      this.conflictIndexSource.next(total - 1);
    } else {
      this.conflictIndexSource.next(index - 1);
    }
  }

  async onNextConflictItem(): Promise<void> {
    this.hasConflictInteraction = true;
    const index = this.conflictItemIndexSource.value;
    const total = await this.totalConflictItems$.pipe(first()).toPromise();

    if (index + 1 === total) {
      this.conflictItemIndexSource.next(0);
    } else {
      this.conflictItemIndexSource.next(index + 1);
    }
  }

  async onPreviousConflictItem(): Promise<void> {
    this.hasConflictInteraction = true;
    const index = this.conflictItemIndexSource.value;
    const total = await this.totalConflictItems$.pipe(first()).toPromise();

    if (index === 0) {
      this.conflictItemIndexSource.next(total - 1);
    } else {
      this.conflictItemIndexSource.next(index - 1);
    }
  }

  onConflictsDismissedChange(dismiss: boolean): void {
    this.conflictsDismissedSource.next(dismiss);
  }

  unallocateItem(item: CalendarItem): void {
    this.roomService.unallocateItem(item);
  }

  onCalendarScroll(scrollPosition: ScrollPosition): void {
    this.calendarScrolledSource.next(scrollPosition);
  }

  /**
   * Called when the sidenav is to be opened.
   * @param item The item being interacted with.
   */
  onViewItem(item: CalendarItem): void {
    this.sidenavItemSource.next(item);
  }

  /**
   * Called when the reservation portal is closed.
   */
  onReservationPortalClosed(): void {
    this.sidenavItemSource.next(null);
  }
}
