import { Injectable } from '@angular/core';
import { BehaviorSubject, Observable } from 'rxjs';
import { first, map } from 'rxjs/operators';
import * as moment from 'moment';
import { Period } from '../../core/filters/period';
import { RcmApiService } from 'resrequest-angular-common';
import { CalendarSortOrder } from '../../core/filters/calendar-sort-order';
import { Property, Accommodation } from 'src/app/shared/components/accommodation-filter/accommodation-filter.component';
import { SearchCriteria } from '../../core/search/search-criteria';
import { SettingsManager } from 'src/app/rooming-calendar/services/settings/settings-manager';
import { CalendarDateBuilder } from '../../core/calendar/calendar-date-builder';

export interface AccommodationMap {
  [accommodationId: string]: AccommodationMapValue;
}
export interface AccommodationMapValue {
  property: Property;
  accommodation: Accommodation;
}

@Injectable()
export class FilterService {
  /**
   * The source for period$.
   */
  private periodSource: BehaviorSubject<Period>;

  /**
   * The period to display for the calendar.
   */
  period$: Observable<Period>;

  /**
   * The source for filterProperties$.
   */
  private filterPropertiesSource: BehaviorSubject<Property[]>;

  /**
   * The available filter properties.
   */
  filterProperties$: Observable<Property[]>;

  /**
   * Map of the available accommodation types to properties.
   */
  accommodationMap$: Observable<AccommodationMap>;

  /**
   * The source for selectedAccommodations$.
   */
  private accommodationsSource: BehaviorSubject<string[]>;

  /**
   * The accommodations that are currently selected.
   */
  accommodations$: Observable<string[]>;

  /**
   * The source for groupCalendarBy$.
   */
  private sortCalendarBySource: BehaviorSubject<CalendarSortOrder>;

  /**
   * How to group the rows of the calendar.
   */
  sortCalendarBy$: Observable<CalendarSortOrder>;

  /**
   * The source for showResName$.
   */
  private showResNameSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the reservation name on a res item or not.
   */
  showResName$: Observable<boolean>;

  /**
   * The source for showResNumber$.
   */
  private showResNumberSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the reservation number on a res item or not.
   */
  showResNumber$: Observable<boolean>;

  /**
   * The source for showGuestNames$.
   */
  private showGuestNameSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the guest names on a res item or not.
   */
  showGuestNames$: Observable<boolean>;

  /**
   * The source for showConflictsOnLoad$.
   */
  private showConflictsOnLoadSource: BehaviorSubject<boolean>;

  /**
   * Whether to show conflicts when the calendar loads.
   */
  showConflictsOnLoad$: Observable<boolean>;

  /**
   * Observable
   */
  private searchTermSource: BehaviorSubject<string>;

  /**
   * Observable
   */
  searchTerm$: Observable<string>;

  /**
   * Observable
   */
  private searchCriteriaSource: BehaviorSubject<SearchCriteria>;

  /**
   * Observable
   */
  searchCriteria$: Observable<SearchCriteria>;

  private settingsManager: SettingsManager;

  /**
   * The source for showRooming$.
   */
  private showRoomingSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the rooming information when the itinerary page
   * is opened or not.
   */
  showRooming$: Observable<boolean>;

  /**
   * The source for showExtras$.
   */
  private showExtrasSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the extras information when the itinerary page
   * is opened or not.
   */
  showExtras$: Observable<boolean>;

  /**
   * The source for expandNotes$.
   */
  private expandNotesSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the expanded notes when the notes page
   * is opened or not.
   */
  expandNotes$: Observable<boolean>;

  /**
   * The source for showAllAccommInd$.
   */
  private showAllAccommIndSource: BehaviorSubject<boolean>;

  /**
   * Indicator for if all properties and accommodations are selected or not
   */
  showAllAccommInd$: Observable<boolean>;

  /**
   * Create the filter service.
   */
  constructor(private api: RcmApiService) {
    this.settingsManager = new SettingsManager();

    const savedDate = this.settingsManager.getLastSetting('start');
    const start = CalendarDateBuilder.momentDate(savedDate);
    this.periodSource = new BehaviorSubject<Period>({
      start,
      days: this.settingsManager.getLastSetting('days')
    });
    this.period$ = this.periodSource.asObservable();
    this.filterPropertiesSource = new BehaviorSubject<Property[]>([]);
    this.filterProperties$ = this.filterPropertiesSource.asObservable();

    this.accommodationMap$ = this.filterProperties$.pipe(map(properties => {
      const accommMap = {};
      properties.forEach(property => {
        property.accommodations.forEach(accommodation => {
          accommMap[accommodation.id] = {
            property,
            accommodation
          };
        });
      });

      return accommMap;
    }));

    this.accommodationsSource = new BehaviorSubject<string[]>(this.settingsManager.getLastSetting('selectedAccommodations'));
    this.accommodations$ = this.accommodationsSource.asObservable();
    this.loadPropertyFilterOptions();
    this.sortCalendarBySource = new BehaviorSubject<CalendarSortOrder>(this.settingsManager.getLastSetting('sortCalendarBy'));
    this.sortCalendarBy$ = this.sortCalendarBySource.asObservable();
    this.showResNameSource = new BehaviorSubject<boolean>(this.settingsManager.getLastSetting('showResName'));
    this.showResName$ = this.showResNameSource.asObservable();
    this.showAllAccommIndSource = new BehaviorSubject<boolean>(true);
    this.showAllAccommInd$ = this.showResNameSource.asObservable();
    this.showResNumberSource = new BehaviorSubject<boolean>(this.settingsManager.getLastSetting('showResNumber'));
    this.showResNumber$ = this.showResNumberSource.asObservable();
    this.showGuestNameSource = new BehaviorSubject<boolean>(this.settingsManager.getLastSetting('showGuestNames'));
    this.showGuestNames$ = this.showGuestNameSource.asObservable();
    this.showConflictsOnLoadSource = new BehaviorSubject<boolean>(this.settingsManager.getLastSetting('showConflictsOnLoad'));
    this.showConflictsOnLoad$ = this.showConflictsOnLoadSource.asObservable();
    this.searchTermSource = new BehaviorSubject<string>('');
    this.searchTerm$ = this.searchTermSource.asObservable();
    this.searchCriteriaSource = new BehaviorSubject<SearchCriteria>({
      guestName: true,
      reservationId: true,
      reservationName: true,
      voucher: true
    });
    this.searchCriteria$ = this.searchCriteriaSource.asObservable();

    this.showRoomingSource = new BehaviorSubject<boolean>(this.settingsManager.getLastSetting('showRooming'));
    this.showRooming$ = this.showRoomingSource.asObservable();
    this.showExtrasSource = new BehaviorSubject<boolean>(this.settingsManager.getLastSetting('showExtras'));
    this.showExtras$ = this.showExtrasSource.asObservable();
    this.expandNotesSource = new BehaviorSubject<boolean>(this.settingsManager.getLastSetting('expandNotes'));
    this.expandNotes$ = this.expandNotesSource.asObservable();
    this.showAllAccommIndSource = new BehaviorSubject<boolean>(true);
    this.showAllAccommInd$ = this.showAllAccommIndSource.asObservable();
  }

  /**
   * Loads the properties and accommodations for the filter.
   */
  loadPropertyFilterOptions(): void {
    this.api.get('/api/v1/filters/get_property_accomm').subscribe((properties: Property[]) => {
      this.filterPropertiesSource.next(properties);

      let accommodations = [];
      let selectedAccommodations = [];
      properties.forEach(property => {
        const ids = property.accommodations.map(accomm => accomm.id);
        accommodations = [...accommodations, ...ids];
      });

      // Apply accommodation filter settings from local storage
      const storedAccommodations = this.settingsManager.getLastSetting('selectedAccommodations');
      if (storedAccommodations.length > 0) {
        storedAccommodations.forEach((accomm: string) => {
          if (accommodations.indexOf(accomm) >= 0) {
            selectedAccommodations.push(accomm);
          }
        });

        // If stored accomms are invalid select all accomms
        if (storedAccommodations.length === 0) {
          selectedAccommodations = accommodations;
        }
      } else {
        // Add all the accommodations as selected for the filter.
        selectedAccommodations = accommodations;
      }

      // Only emit if the selected accommodations have changed
      // from the cached accommodations
      if (storedAccommodations.length !== selectedAccommodations.length) {
        this.accommodationsSource.next(selectedAccommodations);
      } else {
        let changed = true;
        selectedAccommodations.forEach((accomm) => {
          if (storedAccommodations.indexOf(accomm) < 0) {
            changed = false;
          }

          if (changed === false) {
            this.accommodationsSource.next(selectedAccommodations);
          }
        });
      }
      var allAccommSelected = (selectedAccommodations.length != accommodations.length) ? false : true;
      this.onshowAllAccommIndChanged(allAccommSelected);
    });
  }

  /**
   * Called when the start date changes.
   * @param start The new start date.
   */
  onStartDateChanged(start: moment.Moment): void {
    this.period$.pipe(first()).subscribe(period => {
      const newPeriod = { ...period, start };
      this.periodSource.next(newPeriod);
    });
  }

  /**
   * Called when the number of days changes.
   * @param days The number of days to display.
   */
  onDaysChanged(days: number): void {
    this.period$.pipe(first()).subscribe(period => {
      const newPeriod = { ...period, days };
      this.periodSource.next(newPeriod);
    });
  }

  /**
   * Sets the start date back a given number of weeks from the current
   * start date.
   * @param weeks The number of weeks to go back. Defaults to 1.
   */
  backNWeeks(weeks: number = 1): void {
    this.period$.pipe(first()).subscribe(period => {
      const start = period.start.clone().subtract(weeks, 'weeks');
      const newPeriod = { ...period, start };
      this.periodSource.next(newPeriod);
    });
  }

  /**
   * Sets the start date forward a given number of weeks from the current
   * start date.
   * @param weeks The number of weeks to go forward. Defaults to 1.
   */
  forwardNWeeks(weeks: number = 1): void {
    this.period$.pipe(first()).subscribe(period => {
      const start = period.start.clone().add(weeks, 'weeks');
      const newPeriod = { ...period, start };
      this.periodSource.next(newPeriod);
    });
  }

  /**
   * Called when the accommodations have been changed.
   * @param accommodations The new selected accommodation ids.
   */
  onAccommodationsChanged(accommodations: string[], loadPropertyFilterOptions = true): void {
    this.accommodationsSource.next(accommodations);

    if (loadPropertyFilterOptions) {
      this.loadPropertyFilterOptions();
    }
  }

  /**
   * Called when the calendar grouping changes.
   * @param sortOrder The new sort order.
   */
  onSortCalendarByChanged(sortOrder: CalendarSortOrder): void {
    this.sortCalendarBySource.next(sortOrder);
  }

  /**
   * Called when the search term changes.
   * @param term The search term.
   */
  onSearchTermChanged(term: string): void {
    this.searchTermSource.next(term);
  }

  /**
   * Called when the res number show changes.
   * @param show Whether to show the res number.
   */
  onShowResNumberChanged(show: boolean): void {
    this.showResNumberSource.next(show);
  }

  /**
   * Called when the search criteria changes.
   * @param criteria The new search criteria.
   */
  onSearchCriteriaChanged(criteria: SearchCriteria): void {
    this.searchCriteriaSource.next(criteria);
  }

  /**
   * Called when the res name show changes.
   * @param show Whether to show the res name.
   */
  onShowResNameChanged(show: boolean): void {
    this.showResNameSource.next(show);
  }

  /**
   * Called when the selected accommodation types changes.
   * @param show Whether to show the accommodation indicator.
   */
  onshowAllAccommIndChanged(show: boolean): void {
    this.showAllAccommIndSource.next(show);
  }

  /**
   * Called when the guest name show changes.
   * @param show Whether to show the guest names.
   */
  onShowGuestNamesChanged(show: boolean): void {
    this.showGuestNameSource.next(show);
  }

  /**
   * Called when the guest name show changes.
   * @param show Whether to show the conflicts on calendar load.
   */
  onShowConflictsOnLoadChanged(show: boolean): void {
    this.showConflictsOnLoadSource.next(show);
  }

  /**
   * Loads the settings for the given user.
   * @param user The id of the user.
   */
  reloadSettings(user: string): void {
    const savedDate = this.settingsManager.getUserSetting('start', user);
    const start = CalendarDateBuilder.momentDate(savedDate);
    this.periodSource.next({
      start,
      days: this.settingsManager.getUserSetting('days', user)
    });

    this.accommodationsSource.next(this.settingsManager.getUserSetting('selectedAccommodations', user));
    this.sortCalendarBySource.next(this.settingsManager.getUserSetting('sortCalendarBy', user));
    this.showResNameSource.next(this.settingsManager.getUserSetting('showResName', user));
    this.showResNumberSource.next(this.settingsManager.getUserSetting('showResNumber', user));
    this.showGuestNameSource.next(this.settingsManager.getUserSetting('showGuestNames', user));
    this.showConflictsOnLoadSource.next(this.settingsManager.getUserSetting('showConflictsOnLoad', user));

    this.settingsManager.setLastUser(user);
  }

  onShowRoomingChanged(show: boolean): void {
    this.showRoomingSource.next(show);
  }

  onShowExtrasChanged(show: boolean): void {
    this.showExtrasSource.next(show);
  }

  onExpandNotesChanged(show: boolean): void {
    this.expandNotesSource.next(show);
  }
}
