import { Injectable } from '@angular/core';
import { RcmApiService } from 'resrequest-angular-common';
import { CalendarService } from '../../services/calendar/calendar.service';
import { LegacyService } from '../../services/legacy/legacy.service';
import { first } from 'rxjs/operators';
import { RoomService } from '../room/room.service';
import { CalendarItem } from '../../core/calendar/calendar-item';
import { Observable } from 'rxjs';
import { WindowService } from '../window/window.service';

@Injectable()
export class FrontOfHouseService {
  loading$: Observable<boolean>;

  constructor(
    public api: RcmApiService,
    private calendarService: CalendarService,
    private legacyService: LegacyService,
    private roomService: RoomService,
  ) {
    this.loading$ = legacyService.loading$;
  }

  async checkIn(items: CalendarItem[]) {
    items.forEach(item => {
      item.data.roomStatusInd = 5; // Checked in
    });

    try {
      const result = await this.roomService.saveItems(items).toPromise();

      if ('errors' in result) {
        throw new Error(result);
      } else {
        await this.roomService.refreshItems(items);
      }
    } catch (error) {
      console.log(error);
      items.forEach(item => {
        item.data.roomStatusInd = item.savedStatusInd; // Revert status change
      });
    }
  }

  async checkOut(items: CalendarItem[], date?: string) {
    items.forEach(item => {
      item.data.roomStatusInd = 8; // Checked out

      if (date) {
        item.data.roomStatusTime = date;
      }
    });

    try {
      const result = await this.roomService.saveItems(items).toPromise();
      if ('errors' in result) {
        throw new Error(result);
      } else {
        if (date) {
          // Back-dating check outs could cause structural changes to the calendar
          // including items not being present in the current period. So just refresh
          // the whole calendar.
          this.roomService.refresh();
        } else {
          await this.roomService.refreshItems(items);
        }
      }
    } catch (error) {
      console.log(error);
      items.forEach(item => {
        item.data.roomStatusInd = item.savedStatusInd; // Revert status change
        item.data.roomStatusTime = item.savedStatusTime;  // Revert status time change
      });
    }
  }

  manageRooms(item: CalendarItem) {
    this.legacyService.roomFunctions(item.data);
  }

  manageGuests(item: CalendarItem) {
    this.legacyService.addGuests(item.data);
  }

  guestDetails(groupId: string, guestId: string) {
    this.legacyService.guestDetails(groupId, guestId);
  }

  async addExtras() {
    const item = await this.calendarService.activeItem$.pipe(first()).toPromise();
    this.legacyService.addExtras(item.data);
  }

  async manageBilling() {
    const item = await this.calendarService.activeItem$.pipe(first()).toPromise();
    this.legacyService.manageBilling(item.data);
  }

  async auditTrail() {
    const item = await this.calendarService.activeItem$.pipe(first()).toPromise();
    this.legacyService.auditTrail(item.data);
  }

  /**
   * Return items associated with the active item
   */
  async reservationItems(): Promise<CalendarItem[]> {
    const item = await this.calendarService.activeItem$.pipe(first()).toPromise();

    if (item) {
      return this.roomService.itemsByReservationId(item.data.reservationId);
    } else {
      return [];
    }
  }

}
