import { Injectable } from '@angular/core';
import { Observable } from 'rxjs';
import { ReservationContacts } from '../../core/reservation/reservation-contacts';
import { RcmApiService } from 'resrequest-angular-common';
import { ReservationNotes } from '../../core/reservation/reservation-notes';
import { ReservationDetailsRequest } from '../../core/api/requests/reservation-details-request';
import { map } from 'rxjs/operators';
import { Contact } from '../../core/reservation/contact';
import { Itinerary } from '../../core/reservation/itinerary';
import * as moment from 'moment';
import { Extra } from '../../core/reservation/extra';
import { CalendarDateBuilder } from '../../core/calendar/calendar-date-builder';
import { RoomGroup } from '../../core/reservation/room-group';

@Injectable()
export class ReservationService {

  constructor(private api: RcmApiService) { }

  /**
   * Returns the contacts attached to a reservation.
   * @param reservationId The id of the reservation.
   */
  getContacts(reservationId: string): Observable<ReservationContacts> {
    const request: ReservationDetailsRequest = {
      rvReservationIx: [reservationId],
      contactDetailsYn: true
    };

    return this.api.post('/api/v1/reservation/get_details', request).pipe(map((res: any) => {
      const resResponse = res[0];
      const apiAgent = resResponse.contacts.agent;
      let agentName = 'Direct';

      if (apiAgent.persona_id !== '0') {
        agentName = (apiAgent.first_name + ' ' + apiAgent.last_name).trim();
      }

      const agent = {
        id: apiAgent.persona_id,
        name: agentName,
        email: apiAgent.email_address,
        fax: apiAgent.fax_number,
        telephone: apiAgent.tel_number
      };

      const apiContact = resResponse.contacts.contact;
      const contact = {
        id: apiContact.persona_id,
        name: (apiContact.first_name + ' ' + apiContact.last_name).trim(),
        email: apiContact.email_address,
        fax: apiContact.fax_number,
        telephone: apiContact.tel_number
      };

      const apiBillingContact = resResponse.contacts.billing;
      let billingContact: Contact;

      if (apiBillingContact) {
        billingContact = {
          id: apiBillingContact.persona_id,
          name: (apiBillingContact.first_name + ' ' + apiBillingContact.last_name).trim(),
          email: apiBillingContact.web_address,
          fax: apiBillingContact.fax_number,
          telephone: apiBillingContact.tel_number
        };

        return {
          agent,
          contact,
          billingContact
        };
      } else {
        // Doesn't have access to financials.
        return {
          agent,
          contact
        };
      }

    }));
  }

  /**
   * Returns the reservation notes of a reservation.
   * @param reservationId The id of the reservation.
   */
  getReservationNotes(reservationId: string): Observable<ReservationNotes> {
    const request: ReservationDetailsRequest = {
      rvReservationIx: [reservationId],
      notesYn: true
    };

    return this.api.post('/api/v1/reservation/get_details', request).pipe(map((res: any) => {
      const resResponse = res[0];
      const guestNotes = resResponse.rvNoteGuests;
      const reservationNotes = resResponse.rvNoteGeneral;
      const memo = resResponse.rvNoteInternal;
      const nationality = resResponse.rfCountryName;

      return {
        guestNotes,
        reservationNotes,
        memo,
        nationality
      };
    }));
  }

  /**
   * Returns the itinerary of a reservation.
   * @param reservationId The id of the reservation.
   */
  getItinerary(reservationId: string): Observable<Itinerary[]> {
    const request: ReservationDetailsRequest = {
      rvReservationIx: [reservationId],
      itineraryYn: true,
      roomingYn: true,
      extrasYn: true
    };

    return this.api.post('/api/v1/reservation/get_details', request).pipe(map((res: any) => {
      const resResponse = res[0];
      const itinerary = resResponse.itinerary;
      const extras = resResponse.extras;
      const rooming = resResponse.rooming;
      return this.buildUiItinerary(itinerary, extras, rooming);
    }));
  }

  private buildUiItinerary(itinerary: any[], extras: any[], rooming: any): Itinerary[] {
    let buildItems = itinerary.map(itin => {
      return {
        id: itin.rvReservationItemIx,
        arrivalDate: moment(itin.rvItemDateArrive.date),
        departureDate: moment(itin.rvItemDateDepart.date),
        nights: itin.rvItemNights,
        rooms: itin.rvItemAccommCount,
        propertyId: itin.propertyId,
        property: itin.propertyName,
        roomTypeId: itin.acAccommId,
        roomType: itin.acAccommDesc,
        adults: itin.rvItemAdultCount,
        children: itin.rvItemChildCount,
        amount: itin.rvItemAmtNett,
        currency: itin.currencySymbol,
        groups: [],
        extras: []
      };
    });

    // Compare them by arrival date. If they are on the same day then the
    // one with the greater number of nights comes after.
    buildItems = buildItems.sort((item1, item2) => {
      if (item1.arrivalDate.isSame(item2.arrivalDate, 'day')) {
        if (item1.nights < item2.nights) {
          return -1;
        } else if (item1.nights > item2.nights) {
          return 1;
        } else {
          return 0;
        }
      } else if (item1.arrivalDate.isBefore(item2.arrivalDate, 'day')) {
        return -1;
      } else {
        return 1;
      }
    });

    const itemMap = {};

    buildItems.forEach((item, index) => {
      itemMap[item.id] = item;
    });

    let buildExtras = extras.map((extra) => {
      return {
        id: extra.acExtraId,
        name: extra.acExtDesc,
        serviceDate: moment(extra.rvExtraDateServ.date),
        serviceBy: (extra.supplierFirstName + ' ' + extra.supplierLastName).trim(),
        quantity: extra.rvExtraUnits,
        amount: extra.rvExtraAmtNett,
        currencyCode: extra.rfCurrencySymbol,
        reference: extra.rvExtraRef,
        expectedDepartureTime: extra.expectedDepartureTime,
        expectedArrivalTime: extra.expectedArrivalTime,
        note: extra.note,
        internalMemo: extra.memo,
        propertyId: extra.prBusinessId,
        property: extra.propertyName,
        categoryType: extra.categoryType
      };
    });

    buildExtras = buildExtras.sort((extra1, extra2) => {
      if (extra1.serviceDate.isBefore(extra2.serviceDate, 'day')) {
        return -1;
      } else if (extra1.serviceDate.isAfter(extra2.serviceDate, 'day')) {
        return 1;
      } else {
        return 0;
      }
    });

    buildExtras.forEach((extra) => {
      let index = -1;

      for (let i = 0; i < buildItems.length; i++) {
        const item = buildItems[i];

        if (extra.serviceDate.isBetween(item.arrivalDate, item.departureDate, 'day', '[]')) {
          if (index === -1) {
            index = i;

            if (extra.propertyId === item.propertyId) {
              break;
            }
          } else {
            // Multiple possible itineraries overlap so assign it to the first
            // one that has the same property.
            if (extra.propertyId === item.propertyId) {
              index = i;
              break;
            }
          }
        }
      }

      if (index === -1) {
        // Just attach it to the first itinerary.
        index = 0;
      }

      const realExtra = {
        ...extra,
        serviceDate: extra.serviceDate.format(CalendarDateBuilder.DATE_FORMAT)
      };
      buildItems[index].extras.push(realExtra);
    });

    const guestMap = {};

    rooming.guests.forEach(guest => {
      guestMap[guest.prGuestId] = {
        id: guest.prGuestId,
        firstName: guest.prNameFirst,
        lastName: guest.prNameLast
      };
    });

    rooming.groups.forEach(group => {
      const item = itemMap[group.rvReservationItemId];
      const guests = group.guests.map(guest => {
        return guestMap[guest.prGuestId];
      });
      let roomGroup: RoomGroup;

      if (group.acAccommRoomId) {
        const upgraded = group.acAccommTypeId !== item.roomTypeId;
        const upgradedFromId = upgraded ? item.roomTypeId : '';
        const upgradeFromName = upgraded ? item.roomType : '';
        roomGroup = {
          guests,
          roomAllocation: {
            roomId: group.acAccommRoomId,
            roomName: group.roomName,
            accommName: group.acAccommDesc,
            capacity: group.acAccommCapacity,
            upgraded,
            upgradedFromId,
            upgradeFromName
          }
        };
      } else {
        roomGroup = {
          guests
        };
      }

      item.groups.push(roomGroup);
    });

    const realItinerary = buildItems.map(item => {
      return {
        ...item,
        arrivalDate: item.arrivalDate.format(CalendarDateBuilder.DATE_FORMAT),
        departureDate: item.departureDate.format(CalendarDateBuilder.DATE_FORMAT)
      };
    })

    return realItinerary;
  }
}
