import { Injectable, OnDestroy } from '@angular/core';
import { BehaviorSubject, Observable, combineLatest, of, Subject, interval } from 'rxjs';
import { Period } from '../../core/filters/period';
import { Property } from '../../core/room/property';
import { CalendarItem } from '../../core/calendar/calendar-item';
import { RoomDropZone } from '../../core/calendar/room-drop-zone';
import { switchMap, first, map, debounceTime, skipWhile, tap } from 'rxjs/operators';
import { FilterService } from '../filter/filter.service';
import { RcmApiService } from 'resrequest-angular-common';
import * as moment from 'moment';
import { ReservationItem } from '../../core/room/reservation-item';
import { CalendarSortOrder } from '../../core/filters/calendar-sort-order';
import { ItemSplit } from '../../core/room/item-split';
import { PropertyManager } from '../../core/room/property-manager';
import { ChangedManagerList } from '../../core/room/changed-manager-list';
import { BlockItem } from '../../core/room/block-item';
import { CalendarBlockItem } from '../../core/calendar/calendar-block-item';
import { Guid } from 'guid-typescript';
import { SearchCriteria } from '../../core/search/search-criteria';
import { SubSink } from 'subsink';
import { CalendarDateBuilder } from '../../core/calendar/calendar-date-builder';
import { Legend } from '../../core/reservation/legend';
import { ConflictsByDate, Conflict } from '../../core/room/conflict';
import { EXTENSION_DAYS } from '../../core/room/config';
import { InfoService } from 'src/app/shared/services/info/info.service';

@Injectable()
export class RoomService implements OnDestroy {
  /**
   * The source for period$;
   */
  private periodSource: BehaviorSubject<Period>;

  /**
   * The current period for the calendar.
   */
  period$: Observable<Period>;

  /**
   * The source fo sortBy$.
   */
  private sortBySource: BehaviorSubject<CalendarSortOrder>;

  /**
   * The sort order for the calendar.
   */
  sortBy$: Observable<CalendarSortOrder>;

  /**
   * The source for properties$;
   */
  private propertiesSource: BehaviorSubject<Property[]>;

  /**
   * The properties to display on the calendar.
   */
  properties$: Observable<Property[]>;

  /**
   * Observable
   */
  searchTerm$: Observable<string>;

  /**
   * Observable
   */
  searchCriteria$: Observable<SearchCriteria>;
  /**
   * Observable
   */
  searchResults$: Observable<CalendarItem[]>;

  /**
   * The source for items$.
   */
  private itemsSource: BehaviorSubject<CalendarItem[]>;

  /**
   * The items on the extended calendar.
   */
  items$: Observable<CalendarItem[]>;

  /**
   * The items to be displayed on the calendar.
   */
  visibleItems$: Observable<CalendarItem[]>;

  /**
   * The items to display on the calendar
   * sorted by from top left
   */
  sortedItems$: Observable<string[]>;

  /**
   * List of items that match search criteria
   */
  sortedResults$: Observable<string[]>;

  resultIndexSource: BehaviorSubject<number>;
  resultIndex$: Observable<number>;
  totalResults$: Observable<number>;

  /**
   * The source for blockItems$.
   */
  private blockItemsSource: BehaviorSubject<CalendarBlockItem[]>;

  /**
   * The block items for the rooms.
   */
  blockItems$: Observable<CalendarBlockItem[]>;

  /**
   * The block items to display on the calendar.
   */
  visibleBlockItems$: Observable<CalendarBlockItem[]>;

  /**
   * The source for roomDropZones$.
   */
  private roomDropZonesSource: BehaviorSubject<RoomDropZone[]>;

  /**
   * The currently available drop zones in the room rows.
   */
  roomDropZones$: Observable<RoomDropZone[]>;

  /**
   * The property managers for the currently selected properties.
   */
  private propertyManagers: PropertyManager[];

  /**
   * The source for hasRoomingChanges$.
   */
  private hasRoomingChangesSource: BehaviorSubject<boolean>;

  /**
   * Whether there are any unsaved rooming changes or not.
   */
  hasRoomingChanges$: Observable<boolean>;

  /**
   * Maintains the history of managers that had rooming changes.
   */
  private changedManagerLists: ChangedManagerList[];

  /**
   * The source for roomingChanges$.
   */
  private roomingChangesSource: Subject<void>;

  /**
   * Emits updates when rooming changes occur.
   */
  roomingChanges$: Observable<void>;

  /**
   * The source for roomingUpdated$.
   */
  private roomingUpdatedSource: Subject<boolean>;

  /**
   * Notifies when the rooming information has been updated and whether
   * there were any errors.
   */
  roomingUpdated$: Observable<boolean>;

  private subsink: SubSink;

  /**
   * The source for loading$.
   */
  private loadingSource: BehaviorSubject<boolean>;

  /**
   * Emits when the rooming calendar is loading or finished loading
   */
  loading$: Observable<boolean>;

  /**
   * The source for conflictsByDate$.
   */
  private conflictsByDateSource: BehaviorSubject<ConflictsByDate>;

  /**
   * The rooming conflicts grouped by date.
   */
  conflictsByDate$: Observable<ConflictsByDate>;

  /**
   * The list of all conflicts.
   */
  conflicts$: Observable<Conflict[]>;

  /**
   * The source for checkInItems$.
   */
  private checkInItemsSource: BehaviorSubject<CalendarItem[]>;

  /**
   * The items that can be checked in.
   */
  checkInItems$: Observable<CalendarItem[]>;

  /**
   * The total number of items that can be checked in.
   */
  checkInTotal$: Observable<number>;

  /**
   * The source for checkOutItems$.
   */
  private checkOutItemsSource: BehaviorSubject<CalendarItem[]>;

  /**
   * The items that can be checked out.
   */
  checkOutItems$: Observable<CalendarItem[]>;

  /**
   * The total number of items that can be checked out.
   */
  checkOutTotal$: Observable<number>;

  /**
   * Create the room service.
   */
  constructor(
    private filterService: FilterService,
    private api: RcmApiService,
    private infoService: InfoService
  ) {
    this.subsink = new SubSink();
    this.loadingSource = new BehaviorSubject<boolean>(true);
    this.loading$ = this.loadingSource.asObservable();
    this.periodSource = new BehaviorSubject<Period>({
      start: moment(),
      days: 7
    });
    this.period$ = this.periodSource.asObservable();
    this.sortBySource = new BehaviorSubject<CalendarSortOrder>(CalendarSortOrder.Accommmodation);
    this.sortBy$ = this.sortBySource.asObservable();
    this.propertiesSource = new BehaviorSubject<Property[]>([]);
    this.properties$ = this.propertiesSource.asObservable();
    this.itemsSource = new BehaviorSubject<CalendarItem[]>([]);
    this.items$ = this.itemsSource.asObservable();
    this.visibleItems$ = this.items$.pipe(
      map(items => {
        // Only show items that are part of the current period.
        return items.filter(item => item.intersection.intersects);
      })
    );
    this.blockItemsSource = new BehaviorSubject<CalendarBlockItem[]>([]);
    this.blockItems$ = this.blockItemsSource.asObservable();
    this.visibleBlockItems$ = this.blockItems$.pipe(
      map(items => {
        // Only show block items that are part of the current period.
        return items.filter(item => item.intersection.intersects);
      })
    );
    this.roomDropZonesSource = new BehaviorSubject<RoomDropZone[]>([]);
    this.roomDropZones$ = this.roomDropZonesSource.asObservable();
    this.hasRoomingChangesSource = new BehaviorSubject<boolean>(false);
    this.hasRoomingChanges$ = this.hasRoomingChangesSource.asObservable();
    this.propertyManagers = [];
    this.changedManagerLists = [];
    this.roomingChangesSource = new Subject<void>();
    this.roomingChanges$ = this.roomingChangesSource.asObservable();
    this.roomingUpdatedSource = new Subject<boolean>();
    this.roomingUpdated$ = this.roomingUpdatedSource.asObservable();
    this.conflictsByDateSource = new BehaviorSubject<ConflictsByDate>({});
    this.conflictsByDate$ = this.conflictsByDateSource.asObservable();
    this.conflicts$ = this.conflicts();
    this.onConfigurationChange();
    this.roomingUpdatedSource.next(true);
    this.searchTerm$ = this.filterService.searchTerm$.pipe(
      map(term => term.trim())
    );
    this.searchCriteria$ = this.filterService.searchCriteria$;
    this.searchResults$ = combineLatest([this.searchTerm$, this.searchCriteria$, this.visibleItems$]).pipe(
      debounceTime(25),
      map(([term, criteria, items]) => {
        return this.search(term, criteria, items);
      })
    );
    this.sortedItems$ = this.sortedItems();
    this.sortedResults$ = this.sortedResults();
    this.totalResults$ = this.searchResults$.pipe(
      map(results => results.length)
    );
    this.resultIndexSource = new BehaviorSubject<number>(-1);
    this.resultIndex$ = this.resultIndexSource.asObservable();
    this.subsink.sink = this.searchResults$.subscribe((results) => {
      if (results.length > 0) {
        this.resultIndexSource.next(0);
      } else {
        this.resultIndexSource.next(-1);
      }
    });

    this.checkInItemsSource = new BehaviorSubject<CalendarItem[]>([]);
    this.checkInItems$ = this.checkInItemsSource.asObservable();
    this.subsink.sink = this.checkInItems().subscribe(items => {
      this.checkInItemsSource.next(items);
    });
    this.subsink.sink = this.roomingChanges$.subscribe(() => {
      // When changes are made reevaluate items that can be checked in.
      this.checkInItems().pipe(first()).subscribe(items => {
        this.checkInItemsSource.next(items);
      });
    });
    this.checkInTotal$ = this.checkInItems$.pipe(
      map(items => items.length)
    );

    this.checkOutItemsSource = new BehaviorSubject<CalendarItem[]>([]);
    this.checkOutItems$ = this.checkOutItemsSource.asObservable();
    this.subsink.sink = this.checkOutItems().subscribe(items => {
      this.checkOutItemsSource.next(items);
    });
    this.subsink.sink = this.roomingChanges$.subscribe(() => {
      // When changes are made reevaluate items that can be checked out.
      this.checkOutItems().pipe(first()).subscribe(items => {
        this.checkOutItemsSource.next(items);
      });
    });
    this.checkOutTotal$ = this.checkOutItems$.pipe(
      map(items => items.length)
    );
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  /**
   * Responds to filter setting changes.
   */
  private onConfigurationChange(): void {
    this.subsink.sink = combineLatest([
      this.filterService.period$,
      this.filterService.accommodations$,
      this.filterService.sortCalendarBy$,
      this.roomingUpdated$,
      this.infoService.environment$
    ]).pipe(
      tap(() => {
        this.loadingSource.next(true);
      }),
      switchMap(([period, accommodations, sortBy, roomingUpdated, environment]) => {
        return combineLatest([
          this.getRooms(period, accommodations, sortBy),
          this.getItems(period, accommodations),
          of(period),
          of(sortBy),
          of(environment)
        ]);
      })
    ).subscribe(([rooms, items, period, sortBy, environment]) => {
      this.clearRoomingInfo();
      const properties = this.apiRoomsToProperties(rooms);
      this.propertyManagers = this.apiItemsToPropertyManagers(
        items,
        properties,
        period,
        environment
      );
      this.roomDropZonesSource.next([]);
      this.periodSource.next(period);
      this.sortBySource.next(sortBy);
      this.propertiesSource.next(properties);
      this.itemsSource.next(this.calendarItems());
      this.blockItemsSource.next(this.calendarBlockItems());
      this.loadingSource.next(false);
      this.conflictsByDateSource.next(this.conflictsByDate());
    });
  }

  /**
   * Retrieve the rooms for the given configuration.
   * @param period The period to get the rooms for.
   * @param accommodations The accommodations to get rooms for.
   * @param sortBy The grouping and sorting for the calendar.
   */
  private getRooms(
    period: Period,
    accommodations: string[],
    sortOrder: CalendarSortOrder
  ): Observable<any> {
    if (accommodations.length === 0) {
      return of([]);
    }

    const end = period.start.clone().add(period.days - 1, 'days');
    const sortBy = sortOrder === CalendarSortOrder.Room ? 'room' : 'accomm';
    const request = {
      accomm_filter: accommodations,
      period: {
        from: period.start.format('YYYY-MM-DD'),
        to: end.format('YYYY-MM-DD')
      },
      sort_by: sortBy
    };

    return this.api.post('/api/v1/rooming/get_rooms', request);
  }

  /**
   * Retrieve the items for the given configuration.
   * @param period The period to get items for.
   * @param accommodations The accommodations to get items for.
   */
  private getItems(period: Period, accommodations: string[]): Observable<any> {
    if (accommodations.length === 0) {
      return of([]);
    }

    // Get items from extension days before the start date to extension days after the end date of
    // the calendar display.
    const start = period.start.clone().subtract(EXTENSION_DAYS, 'days');
    const end = period.start.clone().add(period.days -1 + EXTENSION_DAYS, 'days');

    const request = {
      accomm_filter: accommodations,
      period: {
        from: start.format('YYYY-MM-DD'),
        to: end.format('YYYY-MM-DD')
      }
    };

    return this.api.post('/api/v1/rooming/get_items', request);
  }

  /**
   * Transforms the properties retrieved from the API into the Property
   * interface.
   */
  private apiRoomsToProperties(rooms: any[]): Property[] {
    const accommodationById = {};

    rooms.forEach((property: any) => {
      (property.accommodation as any[]).forEach(accommodation => {
        accommodationById[accommodation.id] = accommodation;
      });
    });

    const properties: Property[] = rooms.map((property: any) => {
      return {
        id: property.id,
        name: property.name,
        accommodations: property.accommodation.map(accommodation => {
          return {
            id: accommodation.id,
            name: accommodation.name,
            propertyId: property.id
          };
        }),
        rooms: property.rooms.map(room => {
          return {
            id: room.id,
            name: room.name,
            accommodationId: room.accommodation_id,
            accommodationName: accommodationById[room.accommodation_id].name,
            forUnallocated: false
          };
        }),
        showAvailability: false
      };
    });

    return properties;
  }

  /**
   * Transforms the items retrieved
   */
  private apiItemsToPropertyManagers(
    items: { [key: string]: any[] },
    properties: Property[],
    period: Period,
    environment: string
  ): PropertyManager[] {
    const propertyMap = {};
    properties.forEach(property => {
      propertyMap[property.id] = property;
    });

    const propertyManagers: PropertyManager[] = [];

    for (const [key, value] of Object.entries(items)) {
      if (key in propertyMap) {
        const property = propertyMap[key];
        const itemSplit = this.apiItemsToSplitItems(value);
        propertyManagers.push(new PropertyManager(property, itemSplit, period, environment));
      }
    }

    return propertyManagers;
  }

  private apiItemToItemGroup(item: any): ReservationItem {
    const resItem = {
      parentId: item.parent_id,
      id: item.id,
      roomId: item.room_id,
      roomName: item.room_name,
      roomStatusInd: item.room_status_ind,
      roomStatusTime: item.room_status_time,
      currentRoomStatusInd: item.current_room_status_ind,
      requestedYn: item.requested_yn,
      requestedReason: item.reqeusted_reason,
      accommCount: item.accomm_count,
      adultCount: item.adult_count,
      childCount: item.child_count,
      accommId: item.accomm_id,
      accommName: item.accomm_name,
      realAccommId: item.real_accomm_id,
      realAccommName: item.real_accomm_name,
      roomMaxCapacity: item.room_max_capacity,
      propertyId: item.property_id,
      propertyName: item.property_name,
      from: item.from,
      to: item.to,
      toBooked: item.to_booked,
      reservationId: item.reservation_id,
      name: item.name,
      statusId: item.status_id,
      statusName: item.status_name,
      statusProvExpiry: item.status_prov_expiry,
      statusBookingYn: item.status_booking_yn,
      statusCreateExpiry: item.status_create_expiry,
      rateTypeId: item.rate_type_id,
      rateTypeName: item.rate_type_name,
      agentId: item.agent_id,
      agentName: item.agent_name,
      guests: item.guests,
      voucher: item.voucher,
      legend: null,
      accessEnvironment: item.access_environment
    };

    resItem.legend = this.itemLegend(resItem.statusId, resItem.roomStatusInd, resItem.from, resItem.toBooked);

    return resItem;
  }

  /**
   * Tranform the items received from the API into a split of the
   * different types of items.
   */
  private apiItemsToSplitItems(items: any[]): ItemSplit {
    const allocated: ReservationItem[] = [];
    const unallocated: ReservationItem[] = [];
    const blocks: BlockItem[] = [];

    items.forEach(item => {
      if (item.type === 'allocated' || item.type === 'unallocated') {
        const groupItem = this.apiItemToItemGroup(item);

        if (item.type === 'allocated') {
          allocated.push(groupItem);
        } else {
          unallocated.push(groupItem);
        }
      } else if (item.type === 'block') {
        const blockItem = {
          id: item.id,
          parentId: item.parent_id,
          roomId: item.room_id,
          from: item.from,
          to: item.to,
          name: item.name,
          propertyId: item.property_id
        };

        blocks.push(blockItem);
      }
    });

    return {
      allocated,
      unallocated,
      blocks
    };
  }

  /**
   * Returns the legend for the given item.
   * @param item Calendar item to get legend for
   */
  itemLegend(
    tmpReservationStatus: string,
    roomStatus: number, // 2 = not checked in, still awaiting arrival, 5 = checked in, 8 checked out
    tmpAarrivalDate: string,
    tmpDepartureDate: string,
  ): Legend {
    const reservationStatus = parseInt(tmpReservationStatus, 10);
    const arrivalDate = moment(tmpAarrivalDate, CalendarDateBuilder.DATE_FORMAT).startOf('day');
    const departureDate = moment(tmpDepartureDate, CalendarDateBuilder.DATE_FORMAT).startOf('day');
    const today = moment().startOf('day');

    let legend = '';
    let colour = '';

    if (reservationStatus === 25) {
      // In progress
      colour = '#0eb64d';
      legend = 'In progress';
    } else if (today.isBefore(arrivalDate, 'day')) {
      // Future reservation
      colour = '#000';
      legend = 'Future reservation';
    } else if (roomStatus === 2 && arrivalDate.isSame(today, 'day')) {
      // Guests checking in (today)
      colour = '#fbb60f';
      legend = 'Checking in today';
    } else if (
      roomStatus === 5 &&
      today.isBefore(departureDate, 'day') &&
      (today.isSameOrBefore(arrivalDate, 'day') || today.isAfter(arrivalDate, 'day'))
    ) {
      // Guests checked in
      colour = '#2868de';
      legend = 'Checked in';
    } else if (
      roomStatus === 5 &&
      today.isSame(departureDate, 'day')
    ) {
      // Guests checking out (today)
      colour = '#e15b5b';
      legend = 'Checking out today';
    } else if (
      roomStatus === 5 &&
      today.isAfter(departureDate, 'day')
    ) {
      // Late checkout
      colour = '#28cbde';
      legend = 'Late checkout';
    } else if (roomStatus === 8 && today.isBefore(departureDate, 'day')) {
      // Early departure
      colour = '#9255a4';
      legend = 'Early departure';
    } else if (
      roomStatus === 2 &&
      today.isAfter(arrivalDate, 'day') &&
      today.isSameOrBefore(departureDate, 'day')
    ) {
      // Late checkin
      colour = '#b4880e';
      legend = 'Late checkin';
    } else if (
      roomStatus === 8 ||
      roomStatus === 2 &&
      (departureDate.isSame(today, 'day') || today.isAfter(departureDate, 'day'))
    ) {
      // Past reservation
      colour = '#b6b6b6';
      legend = 'Past reservation';
    } else {
      colour = '#fb00ff';
      legend = 'Unknown';
    }

    return { legend, colour };
  }

  /**
   * Retrieves the calendar items from the property managers.
   */
  private calendarItems(): CalendarItem[] {
    let items = [];

    this.propertyManagers.forEach(manager => {
      items = [...items, ...manager.getCalendarItems()];
    });

    return items;
  }

  /**
   * Retrieves the calendar block items from the property managers.
   */
  private calendarBlockItems(): CalendarBlockItem[] {
    let items = [];

    this.propertyManagers.forEach(manager => {
      items = [...items, ...manager.getCalendarBlockItems()];
    });

    return items;
  }

  /**
   * Find possible drop zones for the calendar item.
   */
  findDropZones(item: CalendarItem): void {
    const manager = this.propertyManagerForId(item.propertyId);
    const rooms = manager.availableRooms(item);
    const dropZones = rooms.map(room => {
      return {
        id: Guid.create().toString(),
        propertyId: item.propertyId,
        roomId: room,
        intersection: {...item.intersection},
      };
    });
    this.roomDropZonesSource.next(dropZones);
  }

  /**
   * Retrieves the property manager for the property.
   * @param propertyId The property id.
   */
  private propertyManagerForId(propertyId: string): PropertyManager {
    const manager = this.propertyManagers.find(m => {
      return m.propertyId() === propertyId;
    });
    return manager;
  }

  /**
   * Called when an item has been dropped on a drop zone.
   * @param zone The drop zone where the item was released.
   * @param item The item that was moved.
   */
  itemDropped(zone: RoomDropZone, item: CalendarItem): void {
    const manager = this.propertyManagerForId(zone.propertyId);
    manager.onItemMoved(item, zone.roomId);
    const changedManagerList = { managers: [manager] };
    this.changedManagerLists.push(changedManagerList);
    this.hasRoomingChangesSource.next(true);
    this.roomingChangesSource.next();
  }

  /**
   * Reverts the most recent rooming changes.
   */
  undoLastChange(): void {
    if (this.changedManagerLists.length === 0) {
      return;
    }

    const index = this.changedManagerLists.length - 1;
    const changedManagerList = this.changedManagerLists[index];
    changedManagerList.managers.forEach(manager => {
      manager.undoLastChange();
    });
    this.changedManagerLists.splice(index, 1);

    this.refreshProperties();

    if (this.changedManagerLists.length === 0) {
      this.hasRoomingChangesSource.next(false);
    }

    this.roomingChangesSource.next();
  }

  /**
   * Reverts all rooming changes.
   */
  undoAllChanges(): void {
    if (this.changedManagerLists.length === 0) {
      return;
    }

    const changedManagerLists = this.changedManagerLists.reverse();
    changedManagerLists.forEach(changedManagerList => {
      changedManagerList.managers.forEach(manager => {
        manager.undoLastChange();
      });
    });

    this.refreshProperties();
    this.changedManagerLists = [];
    this.hasRoomingChangesSource.next(false);
    this.roomingChangesSource.next();
  }

  /**
   * Save the rooming changes.
   */
  saveChanges(): void {
    let items = [];
    this.propertyManagers.forEach(manager => {
      items = [...items, ...manager.changedItems()];
    });
    this.loadingSource.next(true);
    this.saveItems(items).subscribe(response => {
      if (response.status) {
        // Items saved successfully.
        this.refresh(true);
      } else {
        // Handle error saving items
        console.log('Error occurred while saving items');
        console.log(response);
        this.refresh(false);
      }
    });
  }

  /**
   * Persists the changes.
   * @param items The items to save.
   */
  saveItems(items: CalendarItem[]): Observable<any> {
    let apiItems = items.map(item => {
      // If it is not roomed then the id of the fake room for unallocated
      // items is useless.
      const roomId = item.isRoomed ? item.roomId : '0';

      const updatedItem = {
        index: item.id,
        res_item_group_id: item.data.id,
        reservation_item_id: item.data.parentId,
      };

      let roomChange = {};

      if (roomId !== item.savedRoomId) {
        roomChange = { room_id: roomId };
      }

      let statusIndChange = {};

      if (item.data.roomStatusInd !== item.savedStatusInd) {
        statusIndChange = { status_ind: item.data.roomStatusInd }
      }

      let statusTimeChange = {};

      if (item.data.roomStatusTime !== item.savedStatusTime) {
        statusTimeChange = { status_time: item.data.roomStatusTime }
      }

      return { ...updatedItem, ...roomChange, ...statusIndChange, ...statusTimeChange };
    });

    // Remove items without changes
    apiItems = apiItems.filter((item: any) => {
      return item.room_id || item.status_ind || item.status_time;
    });

    const request = { rooms: apiItems };

    if (request.rooms.length <= 0) {
      this.refresh();
    } else {
      return this.api.post('/api/v1/rooming/set_items', request);
    }
  }

  /**
   * Refresh roomed items
   * @param items Items to refresh
   */
  async refreshItems(items: CalendarItem[]): Promise<void> {
    const groupIds = items.map(item => item.data.id);
    const accommodations = [];

    // If there is an item without an ID, refresh all items
    let hardRefresh = false;
    items.forEach(item => {
      if (item.data.id  === '') {
        hardRefresh = true;
      }

      const accommId = item.data.realAccommId ? item.data.realAccommId : item.data.accommId;

      if (!accommodations.includes(accommId)) {
        accommodations.push(accommId);
      }
    });

    if (hardRefresh) {
      this.refresh();
      return;
    }

    const period = await this.filterService.period$.pipe(first()).toPromise();
    const response: { [key: string]: any[] } = await this.getItems(period, accommodations).toPromise();
    const apiItems = Object.values(response)
      .reduce((result: any[], value: any[]) => [...result, ...value], []);
    const propertyMap = {};
    const groupItems = apiItems.map(item => this.apiItemToItemGroup(item))
      .filter(item => groupIds.includes(item.id));

    groupItems.forEach(item => {
      const propertyId = item.propertyId;

      if (!(propertyId in propertyMap)) {
        propertyMap[propertyId] = [];
      }

      propertyMap[propertyId].push(item);
    });

    for (const [propertyId, refreshItems] of Object.entries(propertyMap)) {
      const manager = this.propertyManagerForId(propertyId);
      manager.refreshItems(refreshItems as ReservationItem[]);
    }

    this.roomingChangesSource.next();
  }

  /**
   * Clears out previous rooming state.
   */
  private clearRoomingInfo(): void {
    this.changedManagerLists = [];
    this.hasRoomingChangesSource.next(false);
    this.roomingChangesSource.next();
  }

  /**
   * Attempts to room unallocated items.
   */
  autoRoom(): void {
    const changedManagers = [];

    this.propertyManagers.forEach(manager => {
      if (manager.autoRoom()) {
        changedManagers.push(manager);
      }
    });

    this.refreshProperties();

    if (changedManagers.length > 0) {
      this.changedManagerLists.push({
        managers: changedManagers
      });
      this.hasRoomingChangesSource.next(true);
      this.roomingChangesSource.next();
    }
  }

  /**
   * Reemits the same properties so that updated rooms
   * can be reflected.
   */
  private refreshProperties(): void {
    this.properties$.pipe(first()).subscribe(properties => {
      this.propertiesSource.next(properties);
    });
  }

  private search(term: string, criteria: SearchCriteria, items: CalendarItem[]): CalendarItem[] {
    term = term.toLocaleLowerCase();

    if (!term) {
      return [];
    }

    const results = items.filter(item => {
      if (criteria.guestName) {
        const matches = item.data.guests.some(guest => {
          const name = `${guest.firstName} ${guest.lastName}`.toLocaleLowerCase();
          return name.includes(term);
        });

        if (matches) {
          return true;
        }
      }

      if (criteria.reservationId) {
        const id = item.data.reservationId.toLocaleLowerCase();
        if (id.includes(term)) {
          return true;
        }
      }

      if (criteria.reservationName) {
        const name = item.data.name.toLocaleLowerCase();
        if (name.includes(term)) {
          return true;
        }
      }

      if (criteria.voucher) {
        const voucher = item.data.voucher.toLocaleLowerCase();
        if (voucher.includes(term)) {
          return true;
        }
      }
    });

    return results;
  }

  private sortedItems(): Observable<string[]> {
    const timer = interval(200);

    return this.hasRoomingChanges$.pipe(
      switchMap(() => {
        return combineLatest([this.visibleItems$, timer]);
      }),
      map(([items, _]) => {
        return { items, domItems: this.fetchDomItems() };
      }),
      skipWhile(({ items, domItems }) => {
        return items.length !== domItems.length;
      }),
      map(({ domItems }) => {
        return this.sortItems(domItems);
      }),
      map(items => {
        return items.map(element => element.id);
      }),
      first()
    );
  }

  private fetchDomItems(): HTMLElement[] {
    const items = Array.from(document.querySelectorAll('app-reservation-item'));

    return items as HTMLElement[];
  }

  private sortItems(items: HTMLElement[]): HTMLElement[] {
    items.sort((a: HTMLElement, b: HTMLElement) => {
      if (a.offsetTop === b.offsetTop) {
        return a.offsetLeft - b.offsetLeft;
      }

      return a.offsetTop - b.offsetTop;
    });

    return items;
  }

  private sortedResults(): Observable<string[]> {
    return combineLatest([this.searchResults$, this.sortedItems$]).pipe(
      map(([searchResults, sortedItems]) => {
        const ids = {};

        // Get IDs from results
        searchResults.forEach(result => {
          ids[result.id] = result;
        });

        return sortedItems.filter(item => item in ids);
      })
    );
  }

  nextResult(): void {
    combineLatest([this.resultIndex$, this.totalResults$]).pipe(
      first()
    ).subscribe(([resultIndex, totalResults]) => {
      if (totalResults > 0) {
        if (resultIndex === totalResults - 1) {
          this.resultIndexSource.next(0);
        } else {
          this.resultIndexSource.next(resultIndex + 1);
        }
      } else {
        this.resultIndexSource.next(-1);
      }
    });
  }

  previousResult(): void {
    combineLatest([this.resultIndex$, this.totalResults$]).pipe(
      first()
    ).subscribe(([resultIndex, totalResults]) => {
      if (totalResults > 0) {
        if (resultIndex === 0) {
          this.resultIndexSource.next(totalResults - 1);
        } else {
          this.resultIndexSource.next(resultIndex - 1);
        }
      } else {
        this.resultIndexSource.next(-1);
      }
    });
  }

  refresh(success: boolean = true): void {
    this.roomingUpdatedSource.next(success);
  }

  itemsByReservationId(id: string): CalendarItem[] {
    const items = [];

    this.itemsSource.value.forEach((item) => {
      if (item.data.reservationId === id) {
        items.push(item);
      }
    });

    return items;
  }

  itemsByReservationItemId(id: string): CalendarItem[] {
    return this.itemsSource.value.filter(tmpItem => tmpItem.data.parentId === id);
  }

  itemByGroupId(id: string): CalendarItem | null {
    const item = this.itemsSource.value.find(tmpItem => tmpItem.data.id === id);

    if (item) {
      return item;
    } else {
      return null;
    }
  }

  /**
   * Retrieves the calendar items from the property managers.
   */
  private conflictsByDate(): ConflictsByDate {
    const conflicts: ConflictsByDate = {};

    this.propertyManagers.forEach(manager => {
      const temp = manager.getConflicts();

      for (const [date, values] of Object.entries(temp)) {
        if (!(date in conflicts)) {
          conflicts[date] = [];
        }

        conflicts[date] = [...conflicts[date], ...values];
      }
    });

    return conflicts;
  }

  /**
   * Retrieves all the conflicts.
   */
  private conflicts(): Observable<Conflict[]> {
    return this.conflictsByDate$.pipe(
      map(conflictsByDate => {
        let conflicts = [];

        for (const [date, values] of Object.entries(conflictsByDate)) {
          conflicts = [...conflicts, ...values];
        }

        return conflicts;
      })
    );
  }

  unallocateItem(item: CalendarItem): void {
    const manager = this.propertyManagerForId(item.propertyId);
    const hasChanges = manager.moveToUnallocated(item);

    if (hasChanges) {
      this.changedManagerLists.push({
        managers: [manager]
      });
      this.hasRoomingChangesSource.next(true);
      this.roomingChangesSource.next();
    }
  }

  checkInItems(): Observable<CalendarItem[]> {
    return this.visibleItems$.pipe(
      map(items => {
        return items.filter(item => item.canCheckInOut)
          .filter(item => item.data.roomStatusInd === 2 || item.data.roomStatusInd === 8)
          .sort((a, b) => a.data.roomName.localeCompare(b.data.roomName));
      })
    );
  }

  checkOutItems(): Observable<CalendarItem[]> {
    return this.visibleItems$.pipe(
      map(items => {
        return items.filter(item => item.canCheckInOut)
          .filter(item => item.data.roomStatusInd === 5)
          .sort((a, b) => a.data.roomName.localeCompare(b.data.roomName));
      })
    );
  }
}
