import { CalendarSortOrder } from "src/app/rooming-calendar/core/filters/calendar-sort-order";
import * as moment from 'moment';
import { CalendarDateBuilder } from '../../core/calendar/calendar-date-builder';
import { ScrollPosition } from '../../core/calendar/scroll-position';

export interface RoomingSettings {
  selectedAccommodations: string[];
  start: string;
  days: number;
  sortCalendarBy: CalendarSortOrder;
  showGuestNames: boolean;
  showResName: boolean;
  showResNumber: boolean;
  showConflictsOnLoad: boolean;
  showRooming: boolean;
  showExtras: boolean;
  expandNotes: boolean;
}

type UserSettings = {[username: string]: RoomingSettings};

interface CalendarSettings {
  userSettings: UserSettings;
  lastUser: string;
  scrollPosition: ScrollPosition;
}

export class SettingsManager {
  private defaultRoomingSettings: RoomingSettings = {
    selectedAccommodations: [],
    start: CalendarDateBuilder.stringDate(moment()),
    days: 7,
    sortCalendarBy: CalendarSortOrder.Accommmodation,
    showGuestNames: false,
    showResName: true,
    showResNumber: true,
    showConflictsOnLoad: true,
    showRooming: false,
    showExtras: false,
    expandNotes: false
  };

  private defaultCalendarSettings: CalendarSettings = {
    userSettings: {},
    lastUser: '',
    scrollPosition: {xScroll: 0, yScroll: 0}
  };

  constructor() {
    let settings = JSON.parse(localStorage.getItem('settings'));

    if (settings === null) {
      localStorage.setItem('settings', JSON.stringify(this.defaultCalendarSettings));
    } else {
      if (!('userSettings' in settings)) {
        localStorage.setItem('settings', JSON.stringify(this.defaultCalendarSettings));
      }
    }
  }

  private getDefaultSetting(setting: string): any {
    return this.defaultRoomingSettings[setting];
  }

  private getSettings(): CalendarSettings {
    const settings = localStorage.getItem('settings');

    return settings ? JSON.parse(settings) : {};
  }

  setSettings(settings: CalendarSettings): void {
    localStorage.setItem('settings', JSON.stringify(settings));
  }

  private getLastUser(): string {
    return this.getSettings().lastUser;
  }

  setLastUser(user: string): void {
    const savedSettings = this.getSettings();
    savedSettings.lastUser = user;
    this.setSettings(savedSettings);
  }

  isLastUser(user: string): boolean {
    return this.getLastUser() === user;
  }

  setUserSettings(settings: RoomingSettings, user: string) {
      const savedSettings = this.getSettings();
      savedSettings.userSettings[user] = settings;
      savedSettings.lastUser = user;
      this.setSettings(savedSettings);
  }

  getUserSetting(setting: string, user: string): any {
    let settings = this.getSettings();

    if (!(user in settings.userSettings)) {
      // A new setting may have been added.
      const newSettings = {...this.defaultRoomingSettings};
      this.setUserSettings(newSettings, user);
      settings = this.getSettings();
    }

    const userSettings = settings.userSettings[user];

    if (!(setting in userSettings)) {
      // Might be a new setting
      const settingValue = this.getDefaultSetting(setting);
      userSettings[setting] = settingValue;
      this.setUserSettings(userSettings, user);
    }

    return userSettings[setting];
  }

  getLastSetting(setting: string): any {
    const lastUser = this.getLastUser();

    if (lastUser) {
      return this.getUserSetting(setting, lastUser);
    } else {
      return this.getDefaultSetting(setting);
    }
  }

  getScrollPosition(): ScrollPosition {
    let scrollPosition = this.getSettings().scrollPosition;

    if (scrollPosition) {
      return scrollPosition;
    } else {
      return {xScroll: 0, yScroll: 0};
    }
  }

  setScrollPosition(scrollPosition: ScrollPosition): void {
    const settings = this.getSettings();
    settings.scrollPosition = scrollPosition;
    this.setSettings(settings);
  }
}
