import { Injectable, OnDestroy } from '@angular/core';
import { Observable, combineLatest } from 'rxjs';
import { FilterService } from 'src/app/rooming-calendar/services/filter/filter.service';
import { RoomService } from 'src/app/rooming-calendar/services/room/room.service';
import { switchMap, map } from 'rxjs/operators';
import { SettingsManager, RoomingSettings } from './settings-manager';
import { SubSink } from 'subsink';
import { Period } from '../../core/filters/period';
import { InfoService } from 'src/app/shared/services/info/info.service';
import { CalendarDateBuilder } from '../../core/calendar/calendar-date-builder';

@Injectable()
export class SettingsService implements OnDestroy {
  settings$: Observable<RoomingSettings>;
  private subsink: SubSink;

  constructor(
    private roomService: RoomService,
    private filterService: FilterService,
    private infoService: InfoService
  ) {
    this.settings$ = roomService.items$.pipe(
      switchMap(() => combineLatest([
        filterService.accommodations$,
        filterService.period$,
        filterService.sortCalendarBy$,
        filterService.showGuestNames$,
        filterService.showResName$,
        filterService.showResNumber$,
        filterService.showConflictsOnLoad$,
        filterService.showRooming$,
        filterService.showExtras$,
        filterService.expandNotes$
      ])),
      map(([
        selectedAccommodations,
        period,
        sortCalendarBy,
        showGuestNames,
        showResName,
        showResNumber,
        showConflictsOnLoad,
        showRooming,
        showExtras,
        expandNotes
      ]) => {
        const start = CalendarDateBuilder.stringDate((period as Period).start);
        return {
          selectedAccommodations,
          start,
          days: (period as Period).days,
          sortCalendarBy,
          showGuestNames,
          showResName,
          showResNumber,
          showConflictsOnLoad,
          showRooming,
          showExtras,
          expandNotes
        } as RoomingSettings;
      })
    );

    this.subsink = new SubSink();

    const settingsManager = new SettingsManager();

    this.subsink.sink = this.infoService.user$.subscribe(user => {
      if (user && !settingsManager.isLastUser(user.userId)) {
        this.filterService.reloadSettings(user.userId);
      }
    });

    this.subsink.sink = combineLatest([this.settings$, this.infoService.user$]).subscribe(([settings, user]) => {
      if (user && settingsManager.isLastUser(user.userId)) {
        settingsManager.setUserSettings(settings, user.userId);
      }
    });
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }
}
