import { Component, OnInit, Input, Output, EventEmitter, ChangeDetectionStrategy } from '@angular/core';
import { BehaviorSubject, Observable, combineLatest } from 'rxjs';
import { map, first } from 'rxjs/operators';

/**
 * Represents an accommodation.
 */
export interface Accommodation {
  id: string;
  name: string;
  inactive: boolean;
}

/**
 * Represents a property.
 */
export interface Property {
  id: string;
  name: string;
  shortName: string;
  inactive: boolean;
  accommodations: Accommodation[];
}

/**
 * Keeps track of selection state for an accommodation.
 */
interface AccommodationOption {
  id: string;
  name: string;
  inactive: boolean;
  selected: boolean;
}

/**
 * Keeps track of accommodation selection states for properties.
 */
interface PropertyOption {
  id: string;
  name: string;
  inactive: boolean;
  accommodationOptions: AccommodationOption[];
}

@Component({
  selector: 'app-accommodation-filter',
  templateUrl: './accommodation-filter.component.html',
  styleUrls: ['./accommodation-filter.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class AccommodationFilterComponent implements OnInit {
  private propertyOptions: BehaviorSubject<PropertyOption[]>;
  // tslint:disable-next-line: variable-name
  _properties: Property[];

  @Input() set properties (value: Property[]) {
    this._properties = value;
    this.mapProperties(value);
  }

  get properties(): Property[] {
    return this._properties;
  }

  @Input() preselectedAccommodations: string[];
  selectedAccommodations: string[];
  @Input() enableInactiveItemsToggle: boolean;

  @Input() set showInactiveItems(value: boolean) {
    this.showInactiveItems$.next(value);
  }

  propertyTerm$: BehaviorSubject<string>;
  accommodationTerm$: BehaviorSubject<string>;
  showInactiveItems$: BehaviorSubject<boolean>;

  @Output() changed: EventEmitter<string[]>;
  @Input() showTitle: boolean;

  allPropertiesSelected$: Observable<boolean>;
  allPropertiesIndeterminate$: Observable<boolean>;
  showPropertyMap: {[key: string]: Observable<boolean>};
  showAccommodationMap: {[key: string]: Observable<boolean>};
  propertySelectedMap: {[key: string]: Observable<boolean>};
  propertyIndeterminateMap: {[key: string]: Observable<boolean>};
  accommodationSelectedMap: { [key: string]: Observable<boolean> };
  showPropSearchBorder: boolean;
  showAccSearchBorder: boolean;

  constructor() {
    this.propertyOptions = new BehaviorSubject<PropertyOption[]>([]);
    this.preselectedAccommodations = [];
    this.selectedAccommodations = [];
    this.properties = [];
    this.showInactiveItems$ = new BehaviorSubject<boolean>(false);
    this.showInactiveItems = false;
    this.enableInactiveItemsToggle = false;
    this.propertyTerm$ = new BehaviorSubject<string>('');
    this.accommodationTerm$ = new BehaviorSubject<string>('');
    this.changed = new EventEmitter<string[]>();
    this.showTitle = true;

    this.allPropertiesSelected$ = this.allPropertiesSelected();
    this.allPropertiesIndeterminate$ = this.allPropertiesIndeterminate();
    this.showPropertyMap = {};
    this.showAccommodationMap = {};
    this.propertySelectedMap = {};
    this.propertyIndeterminateMap = {};
    this.accommodationSelectedMap = {};
    this.showPropSearchBorder = false;
    this.showAccSearchBorder = false;
  }

  ngOnInit() {
    this.selectedAccommodations = [...this.preselectedAccommodations];

    if (this.properties) {
      this.mapProperties(this.properties);
    }
  }

  private mapProperties(properties: Property[]): void {
    const propertyOptions = properties.map(property => {
      this.showPropertyMap[property.id] = this.showPropertyAndAccommodations(property.id);
      this.propertySelectedMap[property.id] = this.isPropertySelected(property.id);
      this.propertyIndeterminateMap[property.id] = this.isPropertyIndeterminate(property.id);

      return {
        id: property.id,
        name: property.name,
        inactive: property.inactive,
        accommodationOptions: property.accommodations.map(accommodation => {
          const key = `${property.id}_${accommodation.id}`;
          this.showAccommodationMap[key] = this.showAccommodation(property.id, accommodation.id);
          this.accommodationSelectedMap[key] = this.isAccommodationSelected(property.id, accommodation.id);

          return {
            id: accommodation.id,
            name: accommodation.name,
            inactive: accommodation.inactive,
            selected: this.preselectedAccommodations.includes(accommodation.id)
          };
        })
      };
    });
    this.propertyOptions.next(propertyOptions);
  }

  /**
   * Whether an accommodation is selected or not.
   * @param propertyId The property id.
   * @param accommodationId The accommodation id.
   */
  isAccommodationSelected(propertyId: string, accommodationId: string): Observable<boolean> {
    return this.propertyOptions.pipe(
      map(propertyOptions => {
        const propertyOption = propertyOptions.find(p => p.id === propertyId);
        const accommodationOption = propertyOption.accommodationOptions.find(a => a.id === accommodationId);
        return accommodationOption.selected;
      })
    );
  }

  /**
   * Whether a property is selected or not.
   * @param propertyId The property id.
   */
  isPropertySelected(propertyId: string): Observable<boolean> {
    return combineLatest(this.propertyOptions, this.accommodationTerm$, this.showInactiveItems$).pipe(
      map(([propertyOptions, accommodationTerm, showInactiveItems]) => {
        const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
        const propertyOption = propertyOptions.find(p => p.id === propertyId);
        return propertyOption.accommodationOptions
          .filter(a => showInactiveItems || !a.inactive)
          .filter(a => a.name.toLocaleLowerCase().includes(accommodationTermLower))
          .filter(a => !a.selected).length === 0;
      })
    );
  }

  /**
   * Whether the property is indeterminate or not.
   * @param propertyId The property id.
   */
  isPropertyIndeterminate(propertyId: string): Observable<boolean> {
    return combineLatest(this.propertyOptions, this.accommodationTerm$, this.showInactiveItems$).pipe(
      map(([propertyOptions, accommodationTerm, showInactiveItems]) => {
        const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
        const propertyOption = propertyOptions.find(p => p.id === propertyId);
        const hasUnselected = propertyOption.accommodationOptions
          .filter(a => showInactiveItems || !a.inactive)
          .filter(a => a.name.toLocaleLowerCase().includes(accommodationTermLower))
          .filter(a => !a.selected).length > 0;
        const hasSelected = propertyOption.accommodationOptions
          .filter(a => showInactiveItems || !a.inactive)
          .filter(a => a.name.toLocaleLowerCase().includes(accommodationTermLower))
          .filter(a => a.selected).length > 0;
        return hasUnselected && hasSelected;
      })
    );
  }

  /**
   * Whether all properties have been selected or not.
   */
  allPropertiesSelected(): Observable<boolean> {
    return combineLatest(
      this.propertyOptions,
      this.propertyTerm$,
      this.accommodationTerm$,
      this.showInactiveItems$
    ).pipe(
      map(([propertyOptions, propertyTerm, accommodationTerm, showInactiveItems]) => {
        const propertyTermLower = propertyTerm.trim().toLocaleLowerCase();
        const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
        const filteredPropertyOptions = propertyOptions.filter(p => {
          if (!showInactiveItems && p.inactive) {
            return false;
          }

          if (propertyTermLower && !accommodationTermLower) {
            return p.name.toLocaleLowerCase().includes(propertyTermLower);
          } else {
            return true;
          }
        });

        return filteredPropertyOptions.map(p => {
          return p.accommodationOptions
            .filter(a => showInactiveItems || !a.inactive)
            .filter(a => a.name.toLocaleLowerCase().includes(accommodationTermLower))
            .filter(a => !a.selected).length;
        })
        .filter(p => p > 0)
        .length === 0;
      })
    );
  }

  /**
   * Whether all properties are indeterminate or not.
   */
  allPropertiesIndeterminate(): Observable<boolean> {
    return combineLatest(
      this.propertyOptions,
      this.propertyTerm$,
      this.accommodationTerm$,
      this.showInactiveItems$
    ).pipe(
      map(([propertyOptions, propertyTerm, accommodationTerm, showInactiveItems]) => {
        const propertyTermLower = propertyTerm.trim().toLocaleLowerCase();
        const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
        const filteredPropertyOptions = propertyOptions.filter(p => {
          if (!showInactiveItems && p.inactive) {
            return false;
          }

          if (propertyTermLower && !accommodationTermLower) {
            return p.name.toLocaleLowerCase().includes(propertyTermLower);
          } else {
            return true;
          }
        });

        const hasSelectedProperties = filteredPropertyOptions.map(propertyOption => {
          return propertyOption.accommodationOptions
            .filter(a => showInactiveItems || !a.inactive)
            .filter(a => a.name.toLocaleLowerCase().includes(accommodationTermLower))
            .filter(a => a.selected).length > 0;
        })
        .filter(p => p)
        .length > 0;

        const hasUnSelectedProperties = filteredPropertyOptions.map(propertyOption => {
          return propertyOption.accommodationOptions
            .filter(a => showInactiveItems || !a.inactive)
            .filter(a => a.name.toLocaleLowerCase().includes(accommodationTermLower))
            .filter(a => !a.selected).length > 0;
        })
        .filter(p => p)
        .length > 0;

        return hasUnSelectedProperties && hasSelectedProperties;
      })
    );
  }

  /**
   * Called when an acommodation checkbox is clicked on.
   * @param propertyId The property id.
   * @param accommodationId The accommodation id.
   * @param checked Whether the accommodation is checked or not.
   */
  onAccommodationChange(propertyId: string, accommodationId: string, checked: boolean): void {
    this.propertyOptions.pipe(first()).subscribe(options => {
      const propertyOption = options.find(p => p.id === propertyId);
      const accommodationOption = propertyOption.accommodationOptions.find(a => a.id === accommodationId);
      accommodationOption.selected = checked;
      this.updateSelectedAccommodation(accommodationId, checked);
      this.propertyOptions.next(options);
    });
  }

  /**
   * Called when a propertey checkbox is clicked on.
   * @param propertyId The property id.
   * @param checked Whether the property is checked or not.
   */
  onPropertyChange(propertyId: string, checked: boolean): void {
    combineLatest(this.propertyOptions, this.accommodationTerm$, this.showInactiveItems$)
      .pipe(first()).subscribe(([options, accommodationTerm, showInactiveItems]) => {
      const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
      const propertyOption = options.find(p => p.id === propertyId);
      propertyOption.accommodationOptions.forEach(a => {
        if (
          a.name.toLocaleLowerCase().includes(accommodationTermLower)
          && (showInactiveItems || !a.inactive)
        ) {
          a.selected = checked;
          this.updateSelectedAccommodation(a.id, checked);
        }
      });
      this.propertyOptions.next(options);
    });
  }

  /**
   * Called when the master checkbox is clicked on.
   * @param checked Whether the master checkbox is checked or or not.
   */
  onAllChange(checked: boolean): void {
    combineLatest(
      this.propertyOptions,
      this.propertyTerm$,
      this.accommodationTerm$,
      this.showInactiveItems$
    )
    .pipe(first())
    .subscribe(([options, propertyTerm, accommodationTerm, showInactiveItems]) => {
      const propertyTermLower = propertyTerm.trim().toLocaleLowerCase();
      const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
      options.forEach(p => {
        if (!showInactiveItems && p.inactive) {
          // The property is inactive. Ignore.
          return;
        }

        if (propertyTermLower &&  !accommodationTermLower) {
          if (!p.name.toLocaleLowerCase().includes(propertyTermLower)) {
            // The property doesn't match the filter criteria.
            return;
          }
        }

        p.accommodationOptions.forEach(a => {
          if (
            a.name.toLocaleLowerCase().includes(accommodationTermLower)
            && (showInactiveItems || !a.inactive)
          ) {
            a.selected = checked;
            this.updateSelectedAccommodation(a.id, checked);
          }
        });
      });
      this.propertyOptions.next(options);
    });
  }

  /**
   * Updates the list of selected accommodations.
   * @param accommodationId The accommodatin id.
   * @param checked Whether the accommodation is selected or not.
   */
  private updateSelectedAccommodation(accommodationId: string, checked: boolean): void {
    if (checked) {
      if (!this.selectedAccommodations.includes(accommodationId)) {
        this.selectedAccommodations.push(accommodationId);
      }
    } else {
      if (this.selectedAccommodations.includes(accommodationId)) {
        const index = this.selectedAccommodations.findIndex(a => a === accommodationId);
        this.selectedAccommodations.splice(index, 1);
      }
    }
    this.changed.emit(this.selectedAccommodations);
  }

  /**
   * Called when the property search term changes.
   * @param term The new property search term.
   */
  onPropertySearchChange(term: string): void {
    this.propertyTerm$.next(term);
  }

  /**
   * Called when the accommodation search term changes.
   * @param term The new accommodation search term.
   */
  onAccommodationSearchChange(term: string): void {
    this.accommodationTerm$.next(term);
  }

  /**
   * Whether to display the property and its accommodations.
   * If there is an accommodation search term then the property
   * will be shown if there are any matches. If there is only a
   * property search term then the property will only be shown if
   * it matches the term.
   * @param propertyId The id of the property.
   */
  showPropertyAndAccommodations(propertyId: string): Observable<boolean> {
    return combineLatest(this.propertyTerm$, this.accommodationTerm$, this.showInactiveItems$).pipe(
      map(([propertyTerm, accommodationTerm, showInactiveItems]) => {
        const propertyTermLower = propertyTerm.trim().toLocaleLowerCase();
        const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
        const property = this.properties.find(p => p.id === propertyId);

        if (!showInactiveItems && property.inactive) {
          return false;
        }

        if (propertyTermLower) {
          if (!property.name.toLocaleLowerCase().includes(propertyTermLower)) {
            return false;
          }
        }

        if (!accommodationTermLower) {
          return true;
        }

        let hasAccommodationMatch = false;
        property.accommodations.forEach(a => {
          if (
            a.name.toLocaleLowerCase().includes(accommodationTermLower)
            && (showInactiveItems || !a.inactive)
          ) {
            hasAccommodationMatch = true;
          }
        });

        return hasAccommodationMatch;
      })
    );
  }

  /**
   * Whether to display the accommodation or not.
   * @param propertyId The property id.
   * @param accommodationId The accommodation id.
   */
  showAccommodation(propertyId: string, accommodationId: string): Observable<boolean> {
    return combineLatest(this.accommodationTerm$, this.showInactiveItems$).pipe(
      map(([accommodationTerm, showInactiveItems]) => {
        const accommodationTermLower = accommodationTerm.trim().toLocaleLowerCase();
        const property = this.properties.find(p => p.id === propertyId);
        const accommodation = property.accommodations.find(a => a.id === accommodationId);

        if (!showInactiveItems && accommodation.inactive) {
          return false;
        }

        if (!accommodationTermLower) {
          return true;
        }

        return accommodation.name.toLocaleLowerCase().includes(accommodationTermLower);
      })
    );
  }

  /**
   * Called when the show inactive items checkbox is toggled.
   * @param showInactiveItems Whether to show inactive items or not.
   */
  onShowInactiveItemsChange(showInactiveItems: boolean): void {
    this.showInactiveItems$.next(showInactiveItems);
  }
  inputPropFocused(focused: boolean): void {
    this.showPropSearchBorder = focused;
  }
  inputAccFocused(focused: boolean): void {
    this.showAccSearchBorder = focused;
  }
}
