import { Component, OnInit, Output, EventEmitter, OnDestroy, Input, AfterContentInit } from '@angular/core';
import { PortalService } from './portal.service';
import { Observable } from 'rxjs';
import { Portal } from './portal.model';
import { SubSink } from 'subsink';

@Component({
  selector: 'app-portal',
  templateUrl: './portal.component.html',
  styleUrls: ['./portal.component.scss'],
  providers: [PortalService]
})
export class PortalComponent implements OnInit, OnDestroy, AfterContentInit {
  showMenu$: Observable<boolean>;
  activePortal$: Observable<Portal>;
  @Input() startPortal: string;
  @Input() menuOpened: boolean;
  @Input() durablePages: boolean;
  @Input() disableLazyLoading: boolean;
  @Input() enableMenu: boolean;

  @Input() set showLoading(show: boolean) {
    this.portal.onShowLoading(show);
  }

  @Output() menuStateChanged: EventEmitter<boolean>;
  @Output() action: EventEmitter<string>;
  @Output() portalChanged: EventEmitter<Portal>;
  @Output() portalClosed: EventEmitter<void>;
  private subsink: SubSink;

  showLoading$: Observable<boolean>;
  pageLoaded: {[key: string]: boolean};

  constructor(public portal: PortalService) {
    this.showMenu$ = portal.showMenu$;
    this.activePortal$ = portal.activePortal$;
    this.menuOpened = false;
    this.durablePages = false;
    this.disableLazyLoading = false;
    this.menuStateChanged = new EventEmitter();
    this.action = new EventEmitter();
    this.portalChanged = new EventEmitter();
    this.subsink = new SubSink();
    this.portalClosed = new EventEmitter();
    this.pageLoaded = {};
    this.showLoading$ = this.portal.showLoading$;
    this.enableMenu = true;
  }

  ngOnInit() {
    this.subsink.sink = this.portal.showMenu$.subscribe(showMenu => {
      this.menuStateChanged.emit(showMenu);
    });

    this.subsink.sink = this.portal.action$.subscribe(action => {
      this.action.emit(action);
    });

    this.subsink.sink = this.portal.activePortal$.subscribe(portal => {
      this.portalChanged.emit(portal);
    });

    this.subsink.sink = this.portal.close$.subscribe(() => {
      this.portalClosed.emit();
    });

    this.subsink.sink = this.activePortal$.subscribe(portal => {
      if (!portal) {
        return;
      }

      if (!(portal.name in this.pageLoaded)) {
        this.pageLoaded[portal.name] = true;
      }
    });

    if (this.menuOpened) {
      this.portal.onShowMenuChanged(true);
    }
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  getPortalService(): PortalService {
    return this.portal;
  }

  ngAfterContentInit() {
    setTimeout(() => {
      // Run inside setTimeout to avoid content change of checks issue.
      if (this.startPortal) {
        this.portal.onPortalSelected(this.startPortal);
      } else {
        this.portal.selectFirstPortal();
      }
    });
  }
}
