import { Injectable } from '@angular/core';
import { BehaviorSubject, Observable, Subject } from 'rxjs';
import { Portal } from './portal.model';

@Injectable()
export class PortalService {
  /**
   * The source for showMenu$.
   */
  private showMenuSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the menu or not.
   */
  showMenu$: Observable<boolean>;

  /**
   * The source for action$.
   */
  private actionSource: Subject<string>;

  /**
   * The action that was raised when a menu item was clicked.
   */
  action$: Observable<string>;

  /**
   * Map of portal names to portals.
   */
  private portals: {[key: string]: Portal};

  /**
   * The source for activePortal$.
   */
  private activePortalSource: BehaviorSubject<Portal>;

  /**
   * The currently visible portal.
   */
  activePortal$: Observable<Portal>;

  /**
   * The source for close$.
   */
  private closeSource: Subject<void>;

  /**
   * Event raised when the portal should close.
   */
  close$: Observable<void>;

  /**
   * The source for showLoading$.
   */
  private showLoadingSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the loading bar or not.
   */
  showLoading$: Observable<boolean>;

  /**
   * The list of portals.
   */
  portalList: {name: string, portal: Portal}[];

  /**
   * Creates a new instance of the Portal service.
   */
  constructor() {
    this.showMenuSource = new BehaviorSubject<boolean>(false);
    this.showMenu$ = this.showMenuSource.asObservable();
    this.actionSource = new Subject<string>();
    this.action$ = this.actionSource.asObservable();
    this.portals = {};
    this.activePortalSource = new BehaviorSubject<Portal>(null);
    this.activePortal$ = this.activePortalSource.asObservable();
    this.closeSource = new Subject<void>();
    this.close$ = this.closeSource.asObservable();
    this.showLoadingSource = new BehaviorSubject<boolean>(false);
    this.showLoading$ = this.showLoadingSource.asObservable();
    this.portalList = [];
  }

  /**
   * Called when the showMenu option has changed.
   */
  onShowMenuChanged(showMenu: boolean): void {
    this.showMenuSource.next(showMenu);
  }

  /**
   * Called when an action is triggered by a menu item.
   * @param action The action to be raised.
   */
  onActionClicked(action: string): void {
    this.actionSource.next(action);
  }

  /**
   * Adds a portal to the list of portals.
   * @param portal The portal to be added.
   */
  addPortal(portal: Portal): void {
    this.portals[portal.name] = portal;
    this.portalList = Object.keys(this.portals).map(name => {
      return {
        name,
        portal: this.portals[name]
      };
    });
  }

  /**
   * Called when a portal has been selected.
   * @param portalName The name of the portal.
   */
  onPortalSelected(portalName: string): void {
    if (portalName in this.portals) {
      this.showLoadingSource.next(false);
      const portal = this.portals[portalName];
      this.activePortalSource.next(portal);
    }
  }

  /**
   * Called when the portal should close.
   */
  onClose(): void {
    this.closeSource.next();
  }

  /**
   * Tries to show the first available portal.
   */
  selectFirstPortal(): void {
    const keys = Object.keys(this.portals);
    if (keys.length > 0) {
      const key = keys[0];
      this.onPortalSelected(key);
    }
  }

  /**
   * Called when the progress bar should be shown or hidden.
   * @param show Whether to show or hide the progress bar.
   */
  onShowLoading(show: boolean): void {
    // Avoid changing the value of showLoading$ while Angular is still
    // doing change detection.
    setTimeout(() => this.showLoadingSource.next(show));
  }
}
