import { Injectable } from '@angular/core';
import { RcmApiService } from 'resrequest-angular-common';
import { BehaviorSubject, Observable } from 'rxjs';

interface FunctionAccess {
  [functionId: number]: number;
}

interface RestrictionAccess {
  [restriction: string]: boolean;
}

interface Environment {
  isMaster: boolean;
  isProperty: boolean;
  isOffline: boolean;
  isWeb: boolean;
  isPrimaryEnvironmentWeb: boolean;
}

@Injectable()
export class AuthorisationService {
  private functions: FunctionAccess = {};
  private allowedRoutes: string[] = [];
  private environment: Environment;
  private restrictions: RestrictionAccess = {};
  private accessLoadedSource: BehaviorSubject<boolean>;
  accessLoaded$: Observable<boolean>;

  constructor(
    private api: RcmApiService
  ) {
    this.accessLoadedSource = new BehaviorSubject<boolean>(false);
    this.accessLoaded$ = this.accessLoadedSource.asObservable();
    this.loadAuthorisation();
  }

  private loadAuthorisation() {
    const request = {
      accessRequest: {
        functions: [],
        routes: [],
        restrictions: [],
        environment: [
          'isMaster',
          'isProperty',
          'isOffline',
          'isWeb',
          'isPrimaryEnvironmentWeb'
        ]
      },
    };

    this.api.post('/api/v1/authorisation/get_access', request).subscribe((access) => {
      this.functions = access.functions;
      this.allowedRoutes = access.routes;
      this.restrictions = access.restrictions;
      this.environment = access.environment;
      this.accessLoadedSource.next(true);
    });
  }

  isMaster() {
    return this.environment.isMaster;
  }

  isProperty() {
    return this.environment.isProperty;
  }

  isOffline() {
    return this.environment.isOffline;
  }

  isWeb() {
    return this.environment.isWeb;
  }

  isPrimaryEnvironmentWeb() {
    return this.environment.isPrimaryEnvironmentWeb;
  }

  /**
   * Returns true if function access level is 5 or above
   * and false if the function doesn't exist
   */
  isFunctionAllowed(functionId: number): boolean {
    if (functionId in this.functions) {
      return this.functions[functionId] >= 5;
    } else {
      return false;
    }
  }

  functionAccessLevel(functionId: number): number {
    if (functionId in this.functions) {
      return this.functions[functionId];
    } else {
      return 0;
    }
  }
}
