import { Injectable } from '@angular/core';
import { BehaviorSubject, Observable } from 'rxjs';
import { Sidebar } from './sidebar';
import { SidebarOptions } from './sidebar-options';

@Injectable()
export class SidebarService {
  private sidebarSource: BehaviorSubject<Sidebar>;
  sidebar$: Observable<Sidebar>;
  private showSidebarSource: BehaviorSubject<boolean>;
  showSidebar$: Observable<boolean>;
  private hasBackdropSource: BehaviorSubject<boolean>;
  hasBackdrop$: Observable<boolean>;
  private autoFocusSource: BehaviorSubject<boolean>;
  autoFocus$: Observable<boolean>;
  private closeCallback: () => void;

  constructor() {
    this.sidebarSource = new BehaviorSubject<Sidebar>(null);
    this.sidebar$ = this.sidebarSource.asObservable();
    this.showSidebarSource = new BehaviorSubject<boolean>(false);
    this.showSidebar$ = this.showSidebarSource.asObservable();
    this.hasBackdropSource = new BehaviorSubject<boolean>(true);
    this.hasBackdrop$ = this.hasBackdropSource.asObservable();
    this.autoFocusSource = new BehaviorSubject<boolean>(false);
    this.autoFocus$ = this.autoFocusSource.asObservable();
  }

  showSidebar(sidebar: Sidebar, onClose: () => void = null, options: SidebarOptions = null): void {
    const defaultOptions: SidebarOptions = {
      hasBackdrop: true,
      autoFocus: false
    };

    let currentOptions: SidebarOptions;

    if (options === null) {
      options = {};
    }

    currentOptions = {...defaultOptions, ...options};
    this.hasBackdropSource.next(currentOptions.hasBackdrop);
    this.autoFocusSource.next(currentOptions.autoFocus);
    this.sidebarSource.next(sidebar);
    this.showSidebarSource.next(true);
    this.closeCallback = onClose;
  }

  close(): void {
    this.showSidebarSource.next(false);

    if (this.closeCallback) {
      this.closeCallback();
    }
  }

  dispose(): void {
    this.sidebarSource.next(null);
  }
}
