import { Component, OnInit, Input, ViewChildren, QueryList, Output, EventEmitter, ViewEncapsulation } from '@angular/core';
import { FinancialInterfaceSetupService } from './service/financial-interface-setup.service';
import { FinancialOption } from './financial-option';
import { FinancialOptionInputComponent } from './inputs/financial-option-input/financial-option-input.component';
import { RawInputComponent } from './inputs/raw-input/raw-input.component';
import { FinancialOptionInputType } from './financial-option-input-type';

@Component({
  selector: 'app-financial-interface-setup',
  templateUrl: './financial-interface-setup.component.html',
  styleUrls: ['./financial-interface-setup.component.scss'],
  providers: [FinancialInterfaceSetupService],
  encapsulation: ViewEncapsulation.ShadowDom
})
export class FinancialInterfaceSetupComponent implements OnInit {
  @Input() invoicingUnit: string;
  @ViewChildren(FinancialOptionInputComponent) uiInputs: QueryList<FinancialOptionInputComponent>;
  @ViewChildren(RawInputComponent) rawInputs: QueryList<RawInputComponent>;
  @Output() optionsSaved: EventEmitter<void>;
  @Output() closed: EventEmitter<null>;
  selectedIndex: number;
  lastIndex: number;

  constructor(public setupService: FinancialInterfaceSetupService) {
    this.invoicingUnit = '';
    this.closed = new EventEmitter<null>();
    this.selectedIndex = 0;
    this.lastIndex = 0;
    this.optionsSaved = setupService.optionsSaved;
  }

  ngOnInit() {
    this.setupService.loadOptions(this.invoicingUnit);
  }

  syncOptions() {
    // Apply changes from the previously switched out UI to this one.
    let options: FinancialOption[];

    if (this.lastIndex === 1) {
      options = this.uiInputs.map((option: FinancialOptionInputComponent) => {
        return option.financialOption();
      });
    } else {
      options = this.rawInputs.map((option: FinancialOptionInputComponent) => {
        return option.financialOption();
      });
    }

    this.lastIndex = this.selectedIndex;
    this.setupService.syncOptions(options);
  }

  save() {
    // Apply the changes from the current UI to the other.
    let options: FinancialOption[];

    if (this.selectedIndex === 1) {
      options = this.uiInputs.map((option: FinancialOptionInputComponent) => {
        return option.financialOption();
      });
    } else {
      options = this.rawInputs.map((option: FinancialOptionInputComponent) => {
        return option.financialOption();
      });
    }

    this.setupService.syncOptions(options);
    this.setupService.saveOptions();
  }

  cancel() {
    this.closed.emit(null);
    this.setupService.loadOptions(this.invoicingUnit);
  }

  isSelectInput(option: FinancialOption): boolean {
    return option.input.type === FinancialOptionInputType.Select;
  }

  isTextInput(option: FinancialOption): boolean {
    return option.input.type === FinancialOptionInputType.Text;
  }

  isNumericInput(option: FinancialOption): boolean {
    return option.input.type === FinancialOptionInputType.Numeric;
  }

  isCheckboxInput(option: FinancialOption): boolean {
    return option.input.type === FinancialOptionInputType.Checkbox;
  }

  isCsvInput(option: FinancialOption): boolean {
    return option.input.type === FinancialOptionInputType.Csv;
  }

  isSequenceInput(option: FinancialOption): boolean {
    return option.input.type === FinancialOptionInputType.Sequence;
  }

  selectInputValue(option: FinancialOption): string {
    const value = option.input.config.find(x => x.value === option.currentValue);
    if (value) {
      return value.label;
    } else {
      return '';
    }
  }

  checkboxInputValue(option: FinancialOption): string {
    if (!option.currentValue) {
      return '';
    }

    if (option.currentValue === '1') {
      return 'yes';
    } else {
      return 'no';
    }
  }

  csvInputValue(option: FinancialOption): string {
    return option.currentValue;
  }

  sequenceInputValue(option: FinancialOption): string {
    const values = option.currentValue ? option.currentValue.split(',') : [];

    return values.map(x => {
      const config = option.input.config.find(y => y.value === x);
      return config ? config.label : '';
    }).join(',');
  }
}
