import { Injectable, EventEmitter } from '@angular/core';
import { BehaviorSubject, interval, Observable } from 'rxjs';
import { take, first } from 'rxjs/operators';
import { FinancialInterfaceNotification } from '../financial-interface-notification/financial-interface-notification';
import { FinancialInterfaceNotificationType } from '../financial-interface-notification/financial-interface-notification-type';
import { FinancialOption } from '../financial-option';
import { FinancialOptionInput } from '../financial-option-input';
import { FinancialOptionInputType } from '../financial-option-input-type';
import { HttpHeaders } from '@angular/common/http';
import { RcmApiService  } from 'resrequest-angular-common';

const httpOptions = {
    headers: new HttpHeaders({ 'Content-Type': 'application/json' })
};
@Injectable()
export class FinancialInterfaceSetupService {
  invoicingUnit: string;
  isLoading$: BehaviorSubject<boolean>;
  notification$: BehaviorSubject<FinancialInterfaceNotification>;
  options$: BehaviorSubject<FinancialOption[]>;
  uiSynced$: BehaviorSubject<boolean>;
  currentValues: { [key: string]: string};
  baseUrl: string;
  optionsSaved: EventEmitter<void>;

  option1: FinancialOption;
  option2: FinancialOption;
  option3: FinancialOption;
  option4: FinancialOption;
  option5: FinancialOption;
  option6: FinancialOption;

  constructor(private api: RcmApiService) {
    this.invoicingUnit = '';
    this.isLoading$ = new BehaviorSubject<boolean>(false);
    this.notification$ = new BehaviorSubject<FinancialInterfaceNotification>(null);
    this.options$ = new BehaviorSubject<FinancialOption[]>([]);
    this.uiSynced$ = new BehaviorSubject<boolean>(true);
    this.currentValues = {};
    this.baseUrl = '/api/v1/financial_interface';
    this.optionsSaved = new EventEmitter<void>();
  }

  loadOptions(invoicingUnit: string) {
    this.invoicingUnit = invoicingUnit;

    const requestUrl = this.baseUrl + '/get_options';
    const request = {
      invoicingUnit
    };

    this.isLoading$.next(true);
    this.api.post(requestUrl, request).subscribe({
      next: (result: any) => {
        this.isLoading$.next(false);
        this.processLoadedOptions(result);
      },

      error: (err: any) => {
        const action = 'An error occurred while trying to load the options';
        this.isLoading$.next(false);
        this.errorNotification(action, err);
      }
    });
  }

  private processLoadedOptions(data: any): void {
    if ('options' in data) {
      const options = data.options.map((option: any) => {
        return new FinancialOption(
          option.id,
          option.name,
          option.description,
          option.raw_description,
          option.default_value,
          option.current_value,
          this.financialOptionInput(option.input)
        );
      });

      this.options$.next(options);
    } else {
        const action = 'An error occurred while trying to load the options';
        this.errorNotification(action, {});
    }
  }

  private financialOptionInput(input: any): FinancialOptionInput {
    let type: FinancialOptionInputType;

    switch (input.type) {
      case 'select':
        type = FinancialOptionInputType.Select;
        break;
      case 'sequence':
        type = FinancialOptionInputType.Sequence;
        break;
      case 'csv':
        type = FinancialOptionInputType.Csv;
        break;
      case 'numeric':
        type = FinancialOptionInputType.Numeric;
        break;
      case 'checkbox':
        type = FinancialOptionInputType.Checkbox;
        break;
      case 'text':
      default:
        type = FinancialOptionInputType.Text;
        break;
    }

    return new FinancialOptionInput(type, input.config);
  }

  storeCurrentValues(options: FinancialOption[]): void {
    this.currentValues = {};
    options.forEach((option: FinancialOption) => {
      this.currentValues[option.id] = option.currentValue;
    });
  }

  currentValuesChanged(options: FinancialOption[]): boolean {
    let changesFound = false;
    options.forEach((option: FinancialOption) => {
      if (option.id in this.currentValues) {
        if (option.currentValue !== this.currentValues[option.id]) {
          changesFound = true;
        }
      } else {
        changesFound = true;
      }
    });

    return changesFound;
  }

  saveOptions(): void {
    this.isLoading$.next(true);
    this.options$.pipe(take(1)).subscribe((options: FinancialOption[]) => {
      const requestUrl = this.baseUrl + '/save_options';
      const updates = options.map((option: FinancialOption) => {
        return {
          id: option.id,
          value: option.currentValue
        };
      });
      const request = {
        invoicingUnit: this.invoicingUnit,
        options: updates
      };

      this.api.post(requestUrl, request).subscribe({
        next: () => {
          this.successNotification('Options saved successfully');
          this.isLoading$.next(false);
        },

        error: (err: any) => {
          const action = 'An error occurred while trying to save the options';
          this.errorNotification(action, err);
          this.isLoading$.next(false);
        }
      });
    });
  }

  private successNotification(message: string): void {
    const notification = new FinancialInterfaceNotification(
        FinancialInterfaceNotificationType.Success,
        message
      );
    this.notification$.next(notification);

    interval(2000).pipe(first()).subscribe(() => {
      this.clearNotification();
      this.optionsSaved.emit();
    });
  }

  private errorNotification(action: string, error: any): void {
    let message = action;

    if ('message' in error) {
      message += ': ' + error.message;
    } else {
      message += '.';
    }

    const notification = new FinancialInterfaceNotification(
        FinancialInterfaceNotificationType.Error,
        message
      );
    this.notification$.next(notification);
  }

  syncOptions(options: FinancialOption[]): void {
    if (this.currentValuesChanged(options)) {
      this.options$.next(options);
      this.storeCurrentValues(options);
      this.syncUi();
    }
  }

  syncUi(): void {
    this.uiSynced$.next(false);
    setTimeout(() => {
      this.uiSynced$.next(true);
    });
  }

  clearNotification() {
    this.notification$.next(null);
  }
}
