import { Injectable, Inject } from '@angular/core';
import { Observable, BehaviorSubject, interval } from 'rxjs';
import { DOCUMENT } from '@angular/common';
import { Widget } from '../widget';
import { Page } from '../page';
import { filter, take } from 'rxjs/operators';
import { HttpClient, HttpParams } from '@angular/common/http';

@Injectable()
export class SupernovaService {
  private pageInternal$: BehaviorSubject<Page>;
  page$: Observable<Page>;
  supportsFullscreen: boolean;
  isWidgetFullscreen: boolean;
  private loadingInternal$: BehaviorSubject<boolean>;
  loading$: Observable<boolean>;
  private hasErrorsInternal$: BehaviorSubject<boolean>;
  hasErrors$: Observable<boolean>;

  constructor(
    @Inject(DOCUMENT) private document: Document,
    private http: HttpClient
  ) {
    this.pageInternal$ = new BehaviorSubject<Page>(null);
    this.page$ = this.pageInternal$.asObservable().pipe(filter(x => x !== null));
    this.supportsFullscreen = document.fullscreenEnabled;
    this.isWidgetFullscreen = false;
    this.loadingInternal$ = new BehaviorSubject<boolean>(false);
    this.loading$ = this.loadingInternal$.asObservable();
    this.hasErrorsInternal$ = new BehaviorSubject<boolean>(false);
    this.hasErrors$ = this.hasErrorsInternal$.asObservable();

    document.addEventListener('fullscreenchange', () => {
      if (document.fullscreenElement) {
          // fullscreen is activated
          this.isWidgetFullscreen = true;
      } else {
          // fullscreen is cancelled
          this.isWidgetFullscreen = false;
      }
    });
  }

  widgetId(widget: Widget): string {
    return 'resrequest-widget-' + widget.id;
  }

  toggleFullScreen(widget: Widget): void {
    if (!this.isWidgetFullscreen) {
      document.getElementById(this.widgetId(widget)).requestFullscreen();
    } else {
      this.document.exitFullscreen();
    }
  }

  fetchPage(baseUrl: string, principal: string, page: number): void {
    const url = baseUrl + '/principals/' + principal;

    this.loadingInternal$.next(true);
    this.http.get<Page>(url).subscribe({
      next: (result: any) => {
        this.loadingInternal$.next(false);
        if ('pages' in result) {
          const pages = (result.pages as Page[]);
          if (page <= pages.length) {
            this.pageInternal$.next(pages[page - 1]);
          } else {
            this.hasErrorsInternal$.next(true);
          }
        } else {
          this.hasErrorsInternal$.next(true);
        }
      },
      error: () => {
        this.loadingInternal$.next(false);
        this.hasErrorsInternal$.next(true);
      }
    });
  }
}
