#!/bin/bash

trap "exit 1" 10
PROC="$$"

CONFIG_FILE_PATH=config.json
if [ ! -e "${CONFIG_FILE_PATH}" ]
then
    echo "WARNING: Config file \"${CONFIG_FILE_PATH}\" missing. Copy default.config.json to config.json and configure"
    echo "Using default config file \"default.config.json\""
    CONFIG_FILE_PATH="default.config.json"
fi

EXPORT_DIR="export"
DUMP_DIR="${EXPORT_DIR}/dump"
EXPORT_FILE="dump.tar.bz2"
EXPORT_FILE_PATH="${EXPORT_DIR}/${EXPORT_FILE}"
LOG_FILE="$(date "+%F_%H-%M-%S").log"
LOG_DIR=$(jq -c '.logDir' "${CONFIG_FILE_PATH}" --raw-output)
LOG_FILE_PATH="${LOG_DIR}/${LOG_FILE}"

MYSQL_USERNAME=$(jq -c '.database.username' "${CONFIG_FILE_PATH}" --raw-output)
MYSQL_PASSWORD=$(jq -c '.database.password' "${CONFIG_FILE_PATH}" --raw-output)
SERVER_HOST=$(jq -c '.server.host' "${CONFIG_FILE_PATH}" --raw-output)
SERVER_PORT=$(jq -c '.server.port' "${CONFIG_FILE_PATH}" --raw-output)
SERVER_USERNAME=$(jq -c '.server.username' "${CONFIG_FILE_PATH}")
SERVER_PASSWORD=$(jq -c '.server.password' "${CONFIG_FILE_PATH}")

SCRIPT_START_TIME=$(date +%s)
SCRIPT_END_TIME=""

# Setup log file
if [ ! -d "${LOG_DIR}" ]
then
    echo "Creating log directory \""${LOG_DIR}"\""
    mkdir "${LOG_DIR}"
    if [ $? != 0 ]
    then
        echo "Failed to create log directory"
        exit 1
    fi
fi

touch "${LOG_FILE_PATH}"
if [ $? != 0 ]
then
    echo "Failed to create log file"
    exit 1
fi

echo "Logging to \"${LOG_FILE_PATH}\""

log() {
    printf -- "${1}\n"
    log=$(printf -- "[%s] ${1}\n" "$(date "+%F %T")") 
    echo $log >> "${LOG_FILE_PATH}"
}

fatal() {
    log "FATAL ERROR: ${@}" >&2
    kill -10 $PROC
}

db() {
    if [ -z "${MYSQL_PASSWORD}" ]
    then
        mysql -u $MYSQL_USERNAME "$@"
    else
        mysql -u $MYSQL_USERNAME -p$MYSQL_PASSWORD "$@"
    fi
}

dbdump() {
    if [ -z "${MYSQL_PASSWORD}" ]
    then
        mysqldump -u $MYSQL_USERNAME "$@"
    else
        mysqldump -u $MYSQL_USERNAME -p$MYSQL_PASSWORD "$@"
    fi
}

# Setup export directory
if [ -d "${EXPORT_DIR}" ]
then
    log "Cleaning export directory \"${EXPORT_DIR}\""
    rm -Rdf "${EXPORT_DIR}"/*
else
    log "Creating export directory \"${EXPORT_DIR}\""
    mkdir "${EXPORT_DIR}"
fi
log "Creating dump directory \"${DUMP_DIR}\""
mkdir "${DUMP_DIR}"


# Test database server connection
db -e ";"

if [ $? != 0 ]
then
    fatal "Failed to connect to database server"
fi

# List of client IDs in config file
CLIENTS=""
while read CLIENT_ID; do
    CLIENTS=" ${CLIENT_ID}${CLIENTS}"
done <<<$(jq -c '.clients[]' "${CONFIG_FILE_PATH}")

CLIENT_COUNT=$(jq -c '.clients | length' "${CONFIG_FILE_PATH}")
declare -A CLIENTS_NAMES
declare -A CLIENTS_VERSIONS
declare -A CLIENTS_DATABASES

log "+--------------*"
log "| Total clients: ${CLIENT_COUNT}"
log "+---------------"

# Build list of clients
CLIENT_ERROR=false
for CLIENT_ID in $CLIENTS; do
    CLIENT_ID=$(printf "%04d" $CLIENT_ID)
    CLIENT_DATABASE="cn_live_${CLIENT_ID}"

    CLIENT_NAME=$(db censys -s -N -e "SELECT cn_prn_name FROM cn_principal WHERE cn_principal_id = '${CLIENT_ID}'")
    if [ $? != 0 ]
    then
        log "| ${CLIENT_ID} => ERROR: Principal entry missing"
        CLIENT_ERROR=true
        continue
    fi

    RESULT=$(db --skip-column-names -e "SHOW DATABASES LIKE '${CLIENT_DATABASE}'")
    if [ "$RESULT" != "${CLIENT_DATABASE}" ]
    then
        log "| ${CLIENT_ID} => ERROR: Database missing"
        CLIENT_ERROR=true
        continue
    fi

    CLIENT_VERSION=$(db $CLIENT_DATABASE -s -N -e "SELECT rf_db_version_db FROM rf_database LIMIT 1" 2>/dev/null)
    if [ $? != 0 ]
    then
        log "| ${CLIENT_ID} => ERROR: Cannot get client version. Database is likely corrupt"
        CLIENT_ERROR=true
        continue
    fi

    CLIENTS_NAMES["${CLIENT_ID}"]=$CLIENT_NAME
    CLIENTS_VERSIONS["${CLIENT_ID}"]=$CLIENT_VERSION
    CLIENTS_DATABASES["${CLIENT_ID}"]=$CLIENT_DATABASE

    log "| ${CLIENT_ID} => ${CLIENT_NAME} (v${CLIENT_VERSION})"
done
log "+--------------*\n"

if [ $CLIENT_ERROR = true ]
then
    fatal "Incorrect client setup"
fi

# Check if lftp is installed
if ! command -v lftp &> /dev/null
then
    fatal "Missing dependency lftp. Install by running \"apt install lftp\""
    exit
fi

# Test connection to FTP server
log "Testing connection to FTP server (${SERVER_HOST}:${SERVER_PORT})"
RESULT=$(lftp $SERVER_HOST -p $SERVER_PORT <<EOF 2>&1
set ftp:ssl-protect-data true;
set ftp:ssl-force true;
set ssl:verify-certificate no;
user $SERVER_USERNAME $SERVER_PASSWORD;
ls
quit;
EOF
)

if [ $? -eq 0 ]
then
    log "Connected to FTP server successfully\n"
else
    log "Failed to connect to FTP server. Error: ${RESULT}"
fi

# Process client DBs
PROGRESS=1
for CLIENT_ID in $CLIENTS; do
    CLIENT_ID=$(printf "%04d" $CLIENT_ID)
    CLIENT_DATABASE="${CLIENTS_DATABASES[$CLIENT_ID]}"

    DB_SIZE=$(db -s -N -e "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 1) FROM information_schema.tables WHERE table_schema = \"${CLIENT_DATABASE}\";")

    log "+--------------*"
    log "| Client: ${CLIENTS_NAMES[$CLIENT_ID]}"
    log "| ID: ${CLIENT_ID}"
    log "| Database: ${CLIENTS_DATABASES[$CLIENT_ID]}"
    log "| Version: ${CLIENTS_VERSIONS[$CLIENT_ID]}"
    log "| Progress: ${PROGRESS}/${CLIENT_COUNT}"
    log "| Size: ${DB_SIZE} MB"
    log "+--------------*"

    START_TIME=$(date +%s)

    # Create temporary DB
    CLIENT_TMP_DATABASE="zzz_resvega_${CLIENT_DATABASE}"
    log "| [client ${CLIENT_ID}] - Creating temporary database \"${CLIENT_TMP_DATABASE}\""
    db -e "DROP DATABASE IF EXISTS ${CLIENT_TMP_DATABASE}; CREATE DATABASE ${CLIENT_TMP_DATABASE};"

    # Build list of tables to process
    TABLES=""
    while read TABLE_NAME; do
        TABLES="${TABLES}${TABLE_NAME} "
    done <<<$(jq -c '.allowedData[].table' $CONFIG_FILE_PATH --raw-output)

    # Copy tables to new temporary DB and remove triggers
    log "| [client ${CLIENT_ID}] - Copying allowed tables in \"${CLIENT_DATABASE}\" to \"${CLIENT_TMP_DATABASE}\""

    COPY_START_TIME=$(date +%s)
    dbdump $CLIENT_DATABASE $TABLES --skip-triggers | db $CLIENT_TMP_DATABASE
    COPY_END_TIME=$(date +%s)
    COPY_EXECUTION_TIME=$(expr $COPY_END_TIME - $COPY_START_TIME)
    log "| [client ${CLIENT_ID}] - Time to copy tables: ${COPY_EXECUTION_TIME}s"

    # Execute SQL scripts
    log "| [client ${CLIENT_ID}] - Scrubbing sensitive data"

    SQL_SCRIPT_START_TIME=$(date +%s)
    SQL_SCRIPT_RESULT=$(db $CLIENT_TMP_DATABASE < ./cleanup-data.sql 2>&1)

    if [ $? != 0 ]
    then
        fatal "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Failed to scrub data. MySQL response: ${SQL_SCRIPT_RESULT}"
    fi

    SQL_SCRIPT_END_TIME=$(date +%s)
    SQL_SCRIPT_EXECUTION_TIME=$(expr $SQL_SCRIPT_END_TIME - $SQL_SCRIPT_START_TIME)
    log "| [client ${CLIENT_ID}] - Scrubbing completed. Execution time: ${SQL_SCRIPT_EXECUTION_TIME}s"

    # Loop through list of tables and generate SQL to remove columns
    for TABLE_NAME in $TABLES; do
        ALTER_TABLE=false # Whether any columns need to be removed from the table
        TABLE_SQL="ALTER TABLE ${TABLE_NAME} "

        log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Building operations"
        
        TABLE_SIZE=$(db -s -N -e "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 1) FROM information_schema.tables WHERE table_schema = \"${CLIENT_DATABASE}\" AND table_name = \"$TABLE_NAME\";")
        log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Table size ${TABLE_SIZE} MB"
        # Get list of columns in table and remove disallowed columns
        while read COLUMN_NAME; do
            # Check if column is in list of allowed columns
            COLUMN_ALLOWED=$(jq -c ".allowedData[] | select(.table == \"${TABLE_NAME}\") | .columns | index(\"${COLUMN_NAME}\")" $CONFIG_FILE_PATH)
            if [ $COLUMN_ALLOWED == "null" ]
            then
                log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] [operation REMOVE] - ${TABLE_NAME}.${COLUMN_NAME}"

                if [ "$ALTER_TABLE" = true ]
                then
                    TABLE_SQL="${TABLE_SQL}, "
                fi

                TABLE_SQL="${TABLE_SQL} DROP ${COLUMN_NAME}"
                ALTER_TABLE=true;
            else
                log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] [operation PRESERVE] - ${TABLE_NAME}.${COLUMN_NAME}"
                continue
            fi
        done < <(db -s -N -e "SELECT COLUMN_NAME FROM information_schema.columns WHERE TABLE_SCHEMA = '${CLIENT_TMP_DATABASE}' AND table_name = '${TABLE_NAME}'")

        if [ "$ALTER_TABLE" = true ]
        then
            log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Executing operations"
            TABLE_OPERATIONS_START_TIME=$(date +%s)

            ALTER_TABLE_RESULT=$(db $CLIENT_TMP_DATABASE -e "${TABLE_SQL};" 2>&1)

            if [ $? == 0 ]
            then
                log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Operations completed successfully"
                TABLE_OPERATIONS_END_TIME=$(date +%s)
                TABLE_OPERATIONS_EXECUTION_TIME=$(expr $TABLE_OPERATIONS_END_TIME - $TABLE_OPERATIONS_START_TIME)
                log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Table operations execution time: ${TABLE_OPERATIONS_EXECUTION_TIME}s"

                NEW_TABLE_SIZE=$(db -s -N -e "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 1) FROM information_schema.tables WHERE table_schema = \"${CLIENT_TMP_DATABASE}\" AND table_name = \"$TABLE_NAME\";")
                log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Table size after operations ${NEW_TABLE_SIZE} MB. Initial size: ${TABLE_SIZE} MB"
            else
                fatal "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - Failed to complete operations. MySQL response: ${ALTER_TABLE_RESULT}"
            fi
        else
            log "| [client ${CLIENT_ID}] [table ${TABLE_NAME}] - No operations required"
        fi
    done

    # Dump database
    DUMP_FILE_PATH="${DUMP_DIR}/${CLIENT_DATABASE}.sql.bz2"
    log "| [client ${CLIENT_ID}] - Dumping ${CLIENT_TMP_DATABASE} to \"${DUMP_DIR}/${DUMP_FILE_PATH}\""
    dbdump $CLIENT_TMP_DATABASE --skip-triggers | bzip2 > "${DUMP_FILE_PATH}"

    if [ $? == 0 ]
    then
        log "| [client ${CLIENT_ID}] - Dump ${CLIENT_TMP_DATABASE} successful"
    else
        fatal "| [client ${CLIENT_ID}] - Dump ${CLIENT_TMP_DATABASE} failed"
    fi

    TMP_DB_SIZE=$(db -s -N -e "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 1) FROM information_schema.tables WHERE table_schema = \"${CLIENT_TMP_DATABASE}\";")
    DUMP_SIZE=$(du -b "${DUMP_FILE_PATH}" | cut -f1)
    DUMP_SIZE=$(echo "${DUMP_SIZE}" | awk '{ split( "B KB MB GB TB PB" , v ); s=1; while( $1>1024 ){ $1/=1024; s++ } printf "%.2f %s", $1, v[s] }')
    # Remove temporary database
    log "| [client ${CLIENT_ID}] - Removing temporary database \"${CLIENT_TMP_DATABASE}\""
    db -e "DROP DATABASE IF EXISTS ${CLIENT_TMP_DATABASE};"

    END_TIME=$(date +%s)
    EXECUTION_TIME=$(expr $END_TIME - $START_TIME)
    log "| [client ${CLIENT_ID}] - Database size after operations: ${TMP_DB_SIZE} MB. Initial size: ${DB_SIZE} MB"
    log "| [client ${CLIENT_ID}] - Dump size: ${DUMP_SIZE}"
    log "| [client ${CLIENT_ID}] - Execution time: ${EXECUTION_TIME}s"
    log "+--------------*\n"

    PROGRESS=$((PROGRESS+1))
done

# Compress dump files
log "\nCompressing dump files to \"${EXPORT_FILE_PATH}\""
(cd "${DUMP_DIR}"; tar -cjvf ../"${EXPORT_FILE}" * > /dev/null 2>&1)

if [ $? == 0 ]
then
    EXPORT_SIZE=$(du -b "${EXPORT_FILE_PATH}" | cut -f1)
    EXPORT_SIZE=$(echo "${EXPORT_SIZE}" | awk '{ split( "B KB MB GB TB PB" , v ); s=1; while( $1>1024 ){ $1/=1024; s++ } printf "%.2f %s", $1, v[s] }')
    log "Export complete. Size ${EXPORT_SIZE}"
else
    fatal "Failed to compress dump files"
fi

# Check if there is an existing export file on the server
log "Checking for existing dump file on server"
lftp $SERVER_HOST -p $SERVER_PORT <<EOF > /dev/null 2>&1
set ftp:ssl-protect-data true;
set ftp:ssl-force true;
set ssl:verify-certificate no;
user $SERVER_USERNAME $SERVER_PASSWORD;
find "$EXPORT_FILE";
quit;
EOF

if [ $? == 0 ]
then
    log "Existing export file found on server. File will be overwritten"
else
    log "Existing export file not found on server. Continuing"
fi


# Upload export file to server
log "Uploading export file to server"
lftp $SERVER_HOST -p $SERVER_PORT <<EOF > /dev/null 2>&1
set ftp:ssl-protect-data true;
set ftp:ssl-force true;
set ssl:verify-certificate no;
user $SERVER_USERNAME $SERVER_PASSWORD;
put "$EXPORT_FILE_PATH"
quit;
EOF

if [ $? == 0 ]
then
    log "Export file uploaded successfully"
else
    fatal "Failed to upload export file"
fi

SCRIPT_END_TIME=$(date +%s)
EXECUTION_TIME=$(expr $SCRIPT_END_TIME - $SCRIPT_START_TIME)
log "Process finished successfully. Execution time: ${EXECUTION_TIME}s"
